/*
 * $Id: pinger.h,v 1.14 2006/04/20 10:39:32 lorenzo Exp $
 * 
 * Copyright (C) 2006 RIPE NCC
 * 
 * Original developer: Lorenzo Colitti <lorenzo@ripe.net>
 * Contributor(s):
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 */

#include <sys/types.h>
#include <sys/time.h>
#include <time.h>
#include <netinet/in.h>

#define ICMP_NONE -1
#define MAX_RESPONSE_TIME 30

/** One ping response */
struct ping_reply {
	uint8_t type;
	uint8_t code;
	uint16_t seq;
	struct timeval response_time;
};

/** Payload of our ICMP packets */
struct payload {
	/** Timestamp when packet was sent */
	struct timeval tv;
	/** ID of target in target array */
	int targetid;
};

/** Array of targets */
struct targetlist {
	int len;
	struct target *targets;
};

/* Parameters for receive thread */
struct recvparm {
	/** Socket to receive from */
	int sock;
	/** PID, i.e. ICMP ID to filter on */
	pid_t pid;
	/** List of targets */
	struct targetlist targetlist;
};

/** Describes a job to run. A job is a set of pings to one target. */
struct job {
	/** Socket to send to */
	int sock;
	/** Ping target */
	struct target *target;
	/** How many pings to send */
	int numpings;
	/** UDP or ICMP mode */
	int udpmode;
	/** ICMP id to send */
	uint16_t id;
	/** Is this job running? */
	int running;
	/** Which thread is running this job? */
	int threadslot;
	/** Thread */
	pthread_t thread;
	/** Protect against concurrent access */
	pthread_mutex_t lock;
};

/** A ping target */
struct target {
	/** Address of the target. At the moment, IPv4 only to save memory */
	struct sockaddr_in addr;
	/** Position of target in array */
	int id;
	/** Number of packets sent */
	short nsent;
	/** Number of replies received */
	short nreceived;
	/** Replies */
	struct ping_reply *replies;
	/** Protect against concurrent access */
	pthread_mutex_t lock;
};

/** Command-line options */
struct options {
	int numthreads;
	int numpings;
	uint16_t port;
	char *filename;
	struct sockaddr_storage srcaddr;
	char *ifname;
};

#define tv_to_ms(tv) ((tv).tv_sec * 1000 + (tv).tv_usec / 1000.0)

/* pinger.c */
void die(char *str);

/* net.c */
int parse_ip(char *ip, uint16_t port, struct sockaddr *sa, int salen);
struct target *recvpacket(int sock, struct ping_reply *reply, uint16_t id, struct targetlist targetlist);
int bindsocket(int sock, uint16_t id);
int sendudp(int sock, struct target *target, uint16_t id);
int sendping(int sock, struct target *target, uint16_t seq, uint16_t id);
char *ip2str(struct sockaddr *sa);
uint16_t in_cksum(uint8_t *buf, int len);

/* list.c */
struct targetlist readlist(struct options *options);

/* sched.c */
int sleepinterval(int us);
void *send_thread_func(void *args);
void *recv_thread_func(void *args);
int get_next_free_thread(struct job *jobs, int numjobs);

/* util.c */
struct timeval tv_subtract(struct timeval tv1, struct timeval tv2);
struct options getoptions(int argc, char * argv[]);
void hexdump(char *buf, int len);
