/*
 * $Id: pinger.c,v 1.13 2006/04/11 13:48:45 lorenzo Exp $
 * 
 * Copyright (C) 2006 RIPE NCC
 * 
 * Original developer: Lorenzo Colitti <lorenzo@ripe.net>
 * Contributor(s):
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <netinet/ip_icmp.h>
#include <arpa/inet.h>
#include <sys/time.h>
#include <time.h>
#include <unistd.h>
#include <pthread.h>

#include "pinger.h"

void die(char *str) {
	perror(str);
	exit(1);
}

void outputresults(struct targetlist targetlist) {
	struct target *target;
	struct ping_reply *reply;
	int targetid, i;

	for(targetid = 0; targetid < targetlist.len; targetid++) {
		target = targetlist.targets + targetid;
		printf("%s: ", ip2str((struct sockaddr *) &target->addr));
		for(i = 0; i < target->nsent; i++) {
			reply = target->replies + i;
			if(reply->type == ICMP_ECHOREPLY)
				printf("%.2fms ", tv_to_ms(reply->response_time));
			else
				printf("* ");
		}
		printf("\n");
	}
}

int main(int argc, char *argv[]) {
	int sock;
	struct targetlist targetlist;
	struct job *jobs;
	int threadslot, targetid;
	struct recvparm recvparm;
	pid_t pid;
	pthread_t recv_thread;
	struct options options;

	/* Open raw socket */
	sock = socket(PF_INET, SOCK_RAW, IPPROTO_ICMP);
	/* Drop root privileges */
	setuid(getuid());

	/* Parse command-line options */
	options = getoptions(argc, argv);

	/* Fall back to UDP mode if ICMP is not available */
	if(sock < 0 && ! options.port) {
		fprintf(stderr, "root privileges required for ICMP mode. Using UDP mode instead.\n");
		options.port = 53;
	}

	if(options.port) {
		/* UDP mode */
		if(sock >=0)
			close(sock);
		sock = socket(PF_INET, SOCK_DGRAM, IPPROTO_UDP);
	}

	if(sock < 0)
		die("socket");

//	/* Bind socket to ICMP ID */
//	if(bindsocket(sock, pid)) {
//		die("bindsocket");
//	}

	/* Bind local address */
	if(bind(sock, (struct sockaddr *) &options.srcaddr, sizeof(options.srcaddr))) {
		die("bind");
	}

	/* Bind to local interface if requested */
	if(options.ifname) {
		if(setsockopt(sock, SOL_SOCKET, SO_BINDTODEVICE,
		   options.ifname, strlen(options.ifname) + 1)) {
			die("setsockopt");
		}
	}

	/* Store parent pid once and for all */
	pid = getpid();

	
	/* Read list of IPs to ping */
	targetlist = readlist(&options);

	/* Start receive thread */
	recvparm.sock = sock;
	recvparm.pid = pid;
	recvparm.targetlist = targetlist;
	pthread_create(&recv_thread, NULL, recv_thread_func, &recvparm);

	/* Allocate thread array */
	jobs = calloc(options.numthreads, sizeof(struct job));
	if(!jobs)
		die("calloc");

	/* Setup jobs */
	for(threadslot = 0; threadslot < options.numthreads; threadslot++) {
		jobs[threadslot].numpings = options.numpings;
		jobs[threadslot].udpmode = (options.port != 0);
		jobs[threadslot].id = getpid();
		jobs[threadslot].running = 0;
		if(pthread_mutex_init(&jobs[threadslot].lock, NULL)) {
			die("pthread_mutex_init");
		}
	}

	for(targetid = 0; targetid < targetlist.len; targetid++) {
		sleepinterval(1000000 / options.numthreads);
		threadslot = get_next_free_thread(jobs, options.numthreads);

		jobs[threadslot].running = 1;
		jobs[threadslot].threadslot = threadslot;
		jobs[threadslot].sock = sock;
		jobs[threadslot].target = &targetlist.targets[targetid];

		if(pthread_create(&jobs[threadslot].thread, NULL, send_thread_func, &jobs[threadslot]) != 0)
			die("pthread_create");

		if(pthread_detach(jobs[threadslot].thread))
			die("pthread_detach");
	}

	sleep(5);

	outputresults(targetlist);
	return 0;
}
