%{
/*
 mfcalc.y, mfcalc.cpp, Copyright (c) 2004-2006 R.Lackner
 parse string and simple math: based on the bison 'mfcalc' example

    This file is part of RLPlot.

    RLPlot is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    RLPlot is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with RLPlot; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include <math.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <stdio.h>
#include "rlplot.h"

class symrec {
public:
	int type, row, col, a_count;
	unsigned int h_name, h2_name;
	char *name, *text;
	double (*fnctptr)(...);
	symrec *next;
	double var, *a_data;

	symrec(unsigned int h_n, unsigned int h2_n, int typ, symrec *nxt);
	~symrec();
	double GetValue();
	void GetValue(void *res);
	double SetValue(double v);
        void SetValue(void* dest, void* src);
	void SetName(char *nam);
	void InitSS();

private:
	bool isSSval;

};

// syntactical information
struct syntax_info {
	int last_tok;			//resolve ambiguous ':'
	double clval;			//current value for where - clause
	int cl1, cl2;			//limits of clause formula in buffer
	struct syntax_info *next;
	};
static syntax_info *syntax_level = 0L;


typedef struct{
	double  val;
	int type;
	symrec  *tptr;
	double *a_data;
	char *text;
	int a_count;

}YYSTYPE;

static symrec *putsym (unsigned int h_name, unsigned int h2_name, int sym_type);
static symrec *getsym (unsigned int h_name, unsigned int h2_name, char *sym_name = 0L);
static int push(YYSTYPE *res, YYSTYPE *val);
static void store_res(YYSTYPE *res);
static char *PushString(char *text);
static double *PushArray(double *arr);
static double *ReallocArray(double *arr, int size);
static char *add_strings(char *st1, char *st2);
static char *string_value(YYSTYPE *exp);
static double eval(YYSTYPE *sr, YYSTYPE *dst, char* dum);
static int range_array(YYSTYPE * res, char *range);
static int range_array2(YYSTYPE *res1, YYSTYPE *res2);
static void exec_clause(YYSTYPE *res);
static YYSTYPE *proc_clause(YYSTYPE *res);
static void yyerror(char *s);
static void make_time(YYSTYPE *dst, double h, double m, double s);
static int yylex(void);
static double nop() {return 0.0;};

static char res_txt[1000];
static anyResult line_res = {ET_UNKNOWN, 0.0, res_txt};
static DataObj *curr_data;
static char *last_error = 0L;		//error text
static char *last_err_desc = 0L;	//short error description

static char *buffer = 0L;		//the current command buffer
static int buff_pos = 0;
static bool bRecent = false;		//rearrange functions
static int parse_level = 0;		//count reentrances into parser
#define MAX_PARSE 20			//maximum number of reentances 
%}

%token <val>  NUM BOOLVAL STR ARR BLOCK PI E CLVAL PSEP IF ELSE BTRUE BFALSE
%token <val>  DATE1 TIME1 DATETIME1 DIM
%token <tptr> VAR FNCT BFNCT AFNCT SFNCT FUNC1 FUNC2 FUNC3 TXT SRFUNC YYFNC FUNC4
%type  <val>  exp str_exp arr bool

%right  '='
%left	','			/* list separator */
%left	CLAUSE			/* clause (where) operator */
%left	SER
%right	COLR COLC		/* range sep., conditional sep. */
%right	'?'			/* conditional assignment */
%left	AND OR
%left   EQ NE GT GE LT LE
%left   '-' '+'
%left   '*' '/'
%left	'['
%left   NEG 		/* negation-unary minus */
%left	INC DEC		/* increment, decrement */
%left	PINC PDEC	/* pre- increment, decrement */
%left	PDIM		/* dimension array */
%right  '^'	 	/* exponentiation       */

/* Grammar follows */
%%
input:    /* empty string */
	| input line
;

line:	 '\n' | ';' | ','
	| exp '\n'		{store_res(&yyvsp[-1]); return 0;}
	| exp ';'		{store_res(&yyvsp[-1]); return 0;}
	| exp ','		{store_res(&yyvsp[-1]); return 0;}
	| str_exp '\n'		{store_res(&yyvsp[-1]); return 0;}
	| str_exp ';'		{store_res(&yyvsp[-1]); return 0;}
	| error '\n'		{yyerrok;}
;

str_exp:
	STR			{;}
	|str_exp '+' exp	{yyval.text=add_strings(yyvsp[-2].text, string_value(&yyvsp[0])); yyval.type = STR;}
	|exp '+' str_exp	{yyval.text=add_strings(string_value(&yyvsp[-2]), yyvsp[0].text); yyval.type = STR;}
	|str_exp '+' str_exp	{yyval.text=add_strings(yyvsp[-2].text, yyvsp[0].text); yyval.type = STR;}
	|SRFUNC '(' exp ')'	{(($1->fnctptr)(&yyval, &yyvsp[-1], 0L)); yyval.type = STR;}
	|SRFUNC '(' exp PSEP str_exp ')'	{(($1->fnctptr)(&yyval, &yyvsp[-3], yyvsp[-1].text)); yyval.type = STR;}
	|SRFUNC '(' exp PSEP exp ')'		{(($1->fnctptr)(&yyval, &yyvsp[-3], yyvsp[-1].text)); yyval.type = STR;}
	|SRFUNC '(' exp ',' str_exp ')'		{(($1->fnctptr)(&yyval, &yyvsp[-3], yyvsp[-1].text)); yyval.type = STR;}
	|SRFUNC '(' exp ',' exp ')'		{(($1->fnctptr)(&yyval, &yyvsp[-3], yyvsp[-1].text)); yyval.type = STR;}
;

range:
	str_exp			{;}
	|VAR COLR VAR		{if(yyval.text = PushString("                                            ")) 
					sprintf(yyval.text, "%s:%s", $1->name, $3->name); yyval.type = STR;}
;

arr:	ARR
	|exp			{if(!yyval.a_data) {yyval.a_data = PushArray((double*)malloc(sizeof(double))); yyval.a_count = 1; yyval.a_data[0] = yyval.val;}}
	|arr ',' arr		{push(&yyval, &yyvsp[0]);yyval.type = ARR;}
	|arr CLAUSE exp		{exec_clause(&yyval);yyval.type = ARR;} 
	|range			{range_array(&yyval, yyvsp[0].text);yyval.type = ARR;}
	|NUM SER NUM		{if($1 < $3 && (yyval.a_data = PushArray((double*)malloc((int)($3-$1+2)*sizeof(double)))))
					for(yyval.a_count=0; $1<=$3; yyval.a_data[yyval.a_count++] = $1, $1 += 1.0 ); yyval.type = ARR;}
;

bool:	BOOLVAL
	|BFNCT '(' exp ')'	{$$ = (($1->fnctptr)($3)); yyval.type = BOOLVAL;}
	|BTRUE			{$$ = 1.0; yyval.type = BOOLVAL;}
	|BFALSE			{$$ = 0.0; yyval.type = BOOLVAL;}
	|exp AND exp		{$$ = (($1 != 0) && ($3 != 0))? 1 : 0; yyval.type = BOOLVAL;}
	|exp OR exp		{$$ = (($1 != 0) || ($3 != 0))? 1 : 0; yyval.type = BOOLVAL;}
	|exp EQ exp		{$$ = ($1 == $3) ? 1 : 0; yyval.type = BOOLVAL;}
	|exp NE exp		{$$ = ($1 != $3) ? 1 : 0; yyval.type = BOOLVAL;}
	|exp GT exp		{$$ = ($1 > $3) ? 1 : 0; yyval.type = BOOLVAL;}
	|exp GE exp		{$$ = ($1 >= $3) ? 1 : 0; yyval.type = BOOLVAL;}
	|exp LT exp		{$$ = ($1 < $3) ? 1 : 0; yyval.type = BOOLVAL;}
	|exp LE exp		{$$ = ($1 <= $3) ? 1 : 0; yyval.type = BOOLVAL;}
;

exp:	NUM				{$$ = $1; yyval.type = NUM;}
    	|bool				{$$ = $1; yyval.type = BOOLVAL;}
        |TXT				{$$ = 0.0;}
	|CLVAL				{$$ = syntax_level ? syntax_level->clval : 0.0;  yyval.type = NUM;}
	|PI				{$$ = _PI; yyval.type = NUM;}
	|E				{$$ = 2.71828182845905; yyval.type = NUM;}
	|VAR				{$1->GetValue(&yyval);}
	|BLOCK			{eval(&yyvsp[0], &yyval, 0L);}
	|VAR '=' exp		{$1->SetValue(&yyval, &yyvsp[0]);}
	|VAR '=' str_exp	{$1->SetValue(&yyval, &yyvsp[0]);}
	|FNCT '(' exp ')'	{$$ = (($1->fnctptr)($3)); yyval.type = NUM;}
	|AFNCT '(' arr ')'	{$$ = (($1->fnctptr)(proc_clause(&yyvsp[-1]))); yyval.type = NUM;}
	|AFNCT '(' exp PSEP arr ')' { if(!yyval.a_data){yyval.a_data=PushArray((double*)malloc(sizeof(double)));yyval.a_data[0]=$3;yyval.a_count=1;}
		push(&yyval, &yyvsp[-1]);$$ = (($1->fnctptr)(&yyval)); yyval.type = NUM;}
	|AFNCT '(' exp PSEP exp PSEP exp ')' { yyval.a_data = PushArray((double*)malloc(3*sizeof(double)));
		yyval.a_count = 3; yyval.a_data[0] = $3; yyval.a_data[1] = $5; yyval.a_data[2] = $7;	
		$$ = (($1->fnctptr)(&yyval)); yyval.type = NUM;}
	|SFNCT '(' str_exp ')'	{$$ = (($1->fnctptr)(&yyvsp[-1], &yyval, 0L)); yyval.type = NUM;}
	|SFNCT '(' exp ')'	{yyvsp[-1].text = string_value(&yyvsp[-1]); $$ = (($1->fnctptr)(&yyvsp[-1], &yyval, 0L)); yyval.type = NUM;}
	|SFNCT '(' str_exp PSEP str_exp ')' {$$ = (($1->fnctptr)(&yyvsp[-3], &yyval, yyvsp[-1].text)); yyval.type = NUM;}
	|SFNCT '(' exp PSEP str_exp ')' {$$ = (($1->fnctptr)(&yyvsp[-3], &yyval, yyvsp[-1].text)); yyval.type = NUM;}
	|SFNCT '(' exp PSEP exp ')' {$$ = (($1->fnctptr)(&yyvsp[-3], &yyval, yyvsp[-1].text)); yyval.type = NUM;}
	|SFNCT '(' str_exp PSEP exp ')' {$$ = (($1->fnctptr)(&yyvsp[-3], &yyval, yyvsp[-1].text)); yyval.type = NUM;}
	|SFNCT '(' str_exp ',' str_exp ')' {$$ = (($1->fnctptr)(&yyvsp[-3], &yyval, yyvsp[-1].text)); yyval.type = NUM;}
	|SFNCT '(' exp ',' str_exp ')' {$$ = (($1->fnctptr)(&yyvsp[-3], &yyval, yyvsp[-1].text)); yyval.type = NUM;}
	|SFNCT '(' str_exp ',' exp ')' {$$ = (($1->fnctptr)(&yyvsp[-3], &yyval, yyvsp[-1].text)); yyval.type = NUM;}
	|SFNCT '(' exp ',' exp ')' {$$ = (($1->fnctptr)(&yyvsp[-3], &yyval, yyvsp[-1].text)); yyval.type = NUM;}
	|FUNC2 '(' range PSEP range ')' {range_array2(&yyvsp[-3], &yyvsp[-1]);$$ = ((*$1->fnctptr)(&yyvsp[-3], &yyvsp[-1], 0L)); yyval.type = NUM;}
	|FUNC2 '(' range PSEP range PSEP range ')' {range_array2(&yyvsp[-5], &yyvsp[-3]); $$ = ((*$1->fnctptr)(&yyvsp[-5], &yyvsp[-3], yyvsp[-1].text)); yyval.type = NUM;}
	|FUNC3 '(' arr PSEP arr ')'	{$$=((*$1->fnctptr)(proc_clause(&yyvsp[-3]), proc_clause(&yyvsp[-1]), 0L));}
	|FUNC3 '(' arr PSEP arr PSEP range ')' {$$=((*$1->fnctptr)(proc_clause(&yyvsp[-5]), proc_clause(&yyvsp[-3]), yyvsp[-1].text));}
	|FUNC4 '(' exp PSEP exp PSEP arr PSEP range ')' {proc_clause(&yyvsp[-3]); $$=(*$1->fnctptr)($3, $5, &yyvsp[-3], &yyvsp[-1]); yyval.type = NUM;}
	|FUNC1 '(' arr PSEP range ')' {$$ = ((*$1->fnctptr)(proc_clause(&yyvsp[-3]), yyvsp[-1].text));  yyval.type = NUM;}
	|YYFNC '(' ')'		{(*$1->fnctptr)(&yyval, 0L);}
	|YYFNC '(' arr ')'	{(*$1->fnctptr)(&yyval, &yyvsp[-1]);}
	|IF '(' exp ')' BLOCK	{$$ = $3 != 0 ? eval(&yyvsp[0], &yyval, 0L) : 0.0;}
	|IF '(' exp ')' BLOCK ELSE BLOCK	{$$ = $3 != 0.0 ? eval(&yyvsp[-2], &yyval, 0L) : eval(&yyvsp[0], &yyval, 0L);}
	|exp '+' exp		{$$ = $1 + $3; yyval.type = NUM;
				if(yyvsp[0].type == DATE1 || yyvsp[-2].type == DATE1) yyval.type = DATE1;
				else if(yyvsp[0].type == TIME1 || yyvsp[-2].type == TIME1) yyval.type = TIME1;
				else if(yyvsp[0].type == DATETIME1 || yyvsp[-2].type == DATETIME1) yyval.type = DATETIME1;}
	|exp '-' exp		{$$ = $1 - $3; yyval.type = NUM;
				if(yyvsp[0].type == DATE1 || yyvsp[-2].type == DATE1) yyval.type = DATE1;
				else if(yyvsp[0].type == TIME1 || yyvsp[-2].type == TIME1) yyval.type = TIME1;
				else if(yyvsp[0].type == DATETIME1 || yyvsp[-2].type == DATETIME1) yyval.type = DATETIME1;}
	|exp '*' exp		{$$ = $1 * $3; yyval.type = NUM;}
	|exp '/' exp		{yyval.type = NUM; if($3 != 0.0) $$ = $1 / $3;
					else $$ = (getsym(HashValue((unsigned char*)"zdiv"), Hash2((unsigned char*)"zdiv")))->GetValue(); }
	|'-' exp  %prec NEG	{$$ = -$2;}
	|VAR INC		{$$=$1->GetValue(); $1->SetValue($$+1.0); $$ -= 1.0; yyval.type = NUM;}
	|VAR DEC		{$$=$1->GetValue(); $1->SetValue($$-1.0); $$ += 1.0; yyval.type = NUM;}
	|INC VAR %prec PINC	{$$=$2->GetValue(); $2->SetValue($$+1.0); yyval.type = NUM;}
	|DEC VAR %prec PDEC	{$$=$2->GetValue(); $2->SetValue($$-1.0); yyval.type = NUM;}
	|exp '^' exp		{$$ = pow($1, $3);}
	|'(' arr ')'		{memcpy(&yyval, &yyvsp[-1], sizeof(YYSTYPE)); yyvsp[-1].a_data = 0L; yyvsp[-1].a_count = 0;}
	|DIM VAR %prec PDIM '[' exp ']'	{yyval.a_data = PushArray((double*)calloc((int)$4, sizeof(double))); yyval.a_count=(int)($4); 
					yyval.type = ARR; $2->SetValue(&yyval,&yyval);}
	|exp '[' exp ']'	{if(yyvsp[-3].a_data && yyvsp[-1].val >= 0.0 && yyvsp[-1].val < yyvsp[-3].a_count) $$ = yyvsp[-3].a_data[(int)yyvsp[-1].val];
				else {$$ = 0.0; last_err_desc = "#INDEX";} yyval.type = NUM;}
	|exp '[' exp ']' '=' exp {if(yyvsp[-5].a_data && yyvsp[-3].val >= 0.0 && yyvsp[-3].val < yyvsp[-5].a_count) $$ = yyvsp[-5].a_data[(int)yyvsp[-3].val] = $6;
				else {$$ = 0.0; last_err_desc = "#INDEX";} yyval.type = NUM;}
	|NUM COLR NUM COLR NUM	{make_time(&yyval, $1, $3, $5+1.0e-10);}
	|NUM ':' NUM ':' NUM	{make_time(&yyval, $1, $3, $5+1.0e-10);}
	|NUM COLR NUM 		{make_time(&yyval, $1, $3, 1.0e-10);}
	|NUM ':' NUM 		{make_time(&yyval, $1, $3, 1.0e-10);}
	|exp '?' exp COLC exp	{memcpy(&yyval, $1 != 0.0 ? &yyvsp[-2] : &yyvsp[0], sizeof(YYSTYPE))}
	|exp '?' STR COLC STR	{memcpy(&yyval, $1 != 0.0 ? &yyvsp[-2] : &yyvsp[0], sizeof(YYSTYPE))}
	|exp '?' STR COLC exp	{memcpy(&yyval, $1 != 0.0 ? &yyvsp[-2] : &yyvsp[0], sizeof(YYSTYPE))}
	|exp '?' exp COLC STR	{memcpy(&yyval, $1 != 0.0 ? &yyvsp[-2] : &yyvsp[0], sizeof(YYSTYPE))}
;
%%

// The symrec class
symrec::symrec(unsigned int h_n, unsigned int h2_n, int typ, symrec *nxt) 
{
	h_name = h_n;	h2_name = h2_n;		type = typ;
	next = nxt;	row = col = -1;		name = text = 0L;
	var = 0.0;	isSSval = false;
	a_data = 0L;	a_count = 0;
	fnctptr = (double (*)(...))nop;
}

symrec::~symrec()
{
	if(name) free(name);	name = 0L;
	if(text) free(text);	text = 0L;
}

double
symrec::GetValue()
{
	anyResult ares;

	if(isSSval) {
		if(row < 0 && col < 0) InitSS();
		//GetResult( , , ,true) inhibits reentrance into parser !
		if(curr_data->GetResult(&ares, row, col, parse_level > MAX_PARSE)){
			return var = ares.value;
			}
		isSSval = false;
		row = col = -1;
		}
	return var;
}

void
symrec::GetValue(void *re)
{
	anyResult ares;
	YYSTYPE *res = (YYSTYPE*)re;

	if(isSSval) {
		if(row < 0 && col < 0) InitSS();
		res->a_data = 0L;	res->a_count = 0;
		//GetResult( , , ,true) inhibits reentrance into parser !
		if(curr_data->GetResult(&ares, row, col, parse_level > MAX_PARSE)){
			if(text) free(text);		text = 0L;
			switch(ares.type) {
			case ET_VALUE:
				res->type = NUM;	res->val = ares.value;
				res->text = 0L;		break;
			case ET_BOOL:
				res->type = BOOLVAL;	res->val = ares.value;
				res->text = 0L;		break;
			case ET_DATE:
				res->type = DATE1;	res->val = ares.value;
				res->text = 0L;		break;
			case ET_TIME:
				res->type = TIME1;	res->val = ares.value;
				res->text = 0L;		break;
			case ET_DATETIME:
				res->type = DATETIME1;	res->val = ares.value;
				res->text = 0L;		break;
			case ET_TEXT:
				res->type = STR;	res->val = 0.0;
				if(ares.text) res->text = PushString(text = strdup(ares.text));
				else res->text = 0L;	break;
			default:
				res->type = NUM;	res->val = var;
				res->text = 0L;		break;
				}
			var = res->val;
			return;
			}
		isSSval = false;
		row = col = -1;
		}
	if(a_data && a_count) {
		res->a_data = a_data;	res->a_count = a_count;
		res->val = 0.0;		res->type = ARR;
		}
	else if(text && text[0]) {
		res->text = PushString(text);
		res->val = var;		res->type = STR;
		}
	else {
		res->type = NUM;	res->val = var;
		res->text = 0L;
		}
}

double 
symrec::SetValue(double v)
{
	if(isSSval) {
		if(row < 0 && col < 0) InitSS();
		if(curr_data->SetValue(row, col, v)){
			if(curr_data) curr_data->Command(CMD_UPDATE, 0L, 0L);
			return var = v;
			}
		isSSval = false;
		row = col = -1;
		}
	a_data = 0L;	a_count = 0;
	return var = v;
}

void 
symrec::SetValue(void* d, void* s)
{
	YYSTYPE *dest = (YYSTYPE*)d;
	YYSTYPE *src = (YYSTYPE*)s;

	if(isSSval && curr_data) {
		if(row < 0 && col < 0) InitSS();
		if(last_err_desc) curr_data->SetText(row, col, last_err_desc);
		else if(src->type == STR) curr_data->SetText(row, col, src->text);
		else if(src->type == ARR || (src->a_data)) curr_data->SetText(row, col, "#ARRAY");
		else if(src->type == VAR && src->tptr->type == TXT) curr_data->SetText(row, col, src->tptr->text);
		else curr_data->SetValue(row, col, src->val);
		curr_data->Command(CMD_UPDATE, 0L, 0L);
		}
	var = src->val;
	if(text) free(text);		text = 0L;
	if(src->text && src->text[0]) 	text = strdup(src->text);
	a_data = src->a_data;		a_count = src->a_count;
	GetValue(d);
	return;
}

void
symrec::SetName(char *nam)
{
	if(name || !nam || !nam[0]) return;
	name = strdup(nam);
	if((name && curr_data) && (isalpha(name[0]) || name[0] == '$') && isdigit(name[strlen(name)-1])) isSSval=true;
}

void
symrec::InitSS()
{
	AccRange *ar;

	if(row<0 && col<0 &&(ar = new AccRange(name))) {
		ar->GetFirst(&col, &row);
		delete(ar);
		}
}

static void yyerror(char *s)
{  
	//called by yyparse on error
	if(curr_data) curr_data->Command(CMD_ERROR, last_error = s, 0L);
	else printf("%s\n", s);
}

static void yyargserr(char *s)
{
	//call from function on argument type/number mismatch
	yyerror(s);
	last_err_desc = "#ARGS";
}

static void make_time(YYSTYPE *dst, double h, double m, double s)
{
	if(!dst || h < 0.0 || 24.0 < h || m < 0.0 || 60.0 < m || s < 0.0 || 60.0 < s) {
		yyerror("parse error");			return;
		}
	dst->val = s/60.0 + m;		dst->val = dst->val/60.0 + h;
	dst->val /= 24.0;		dst->type = TIME1;
}

static void store_res(YYSTYPE *res)
{
	if(last_err_desc) {
		line_res.type = ET_TEXT;
		line_res.value = 0.0;
		strcpy(res_txt, last_err_desc);
		}
	else if(res->type == NUM){
		line_res.type = ET_VALUE;
		line_res.value = res->val;
		}
	else if(res->type == BOOLVAL){
		line_res.type = ET_BOOL;
		line_res.value = res->val;
		}
	else if(res->type == DATE1){
		line_res.type = ET_DATE;
		line_res.value = res->val;
		}
	else if(res->type == TIME1){
		line_res.type = ET_TIME;
		line_res.value = res->val;
		}
	else if(res->type == DATETIME1){
		line_res.type = ET_DATETIME;
		line_res.value = res->val;
		}
 	else if(res->type == STR) {
		line_res.type = ET_TEXT;
		line_res.value = 0.0;
		if(res->text) strcpy(res_txt, res->text);
		}
	else if((res->type == ARR || (res->a_data)) && res->a_count == 1) {
		line_res.type = ET_VALUE;
		line_res.value = res->a_data[0];
		}
	else if(res->type == ARR && !(res->a_data) && !(res->a_count)) {
		line_res.type = ET_VALUE;
		line_res.value = res->val;
		}
	else if(res->type == ARR || (res->a_data)) {
		line_res.type = ET_TEXT;
		line_res.value = 0.0;
		strcpy(res_txt, "#ARRAY");
		}
	else if(res->tptr && res->tptr->type == TXT) {
		line_res.type = ET_TEXT;
		line_res.value = 0.0;
		if(res->tptr->text) strcpy(res_txt, res->tptr->text);
		}
	else {
		line_res.type = ET_VALUE;
		line_res.value = res->val;
		}
}

static char *add_strings(char *st1, char *st2)
{
	char *newstr, *ret;

	if(st1 && st2) {
		if(newstr = (char*)malloc(strlen(st1) +strlen(st2) +4)) {
			sprintf(newstr, "%s%s", st1, st2);
			ret = PushString(newstr);
			free(newstr);
			return ret;
			}
		else return 0L;
		}
	if(st1) return st1;
	if(st2) return st2;
	return 0L;
}

static char *string_value(YYSTYPE *exp)
{
	char *st1, tmp[50];

	if(exp->type == STR){
		st1 = exp->text;
		}
	else if(exp->tptr && exp->tptr->type == TXT) {
		st1 = exp->tptr->text;
		}
	else {
		sprintf(tmp,"%g", exp->val);
		st1 = tmp;
		}
	return PushString(st1);
}

// store syntactical information
static void push_syntax()
{
	syntax_info *next;

	if(!(next = (syntax_info*)calloc(1, sizeof(syntax_info)))) return;
	if(syntax_level)memcpy(next, syntax_level, sizeof(syntax_info));
	next->next=syntax_level;
	syntax_level = next;
}

static void pop_syntax()
{
	syntax_info *si;

	if(si = syntax_level) {
		syntax_level = si->next;
		free(si);
		}
}

static double eval(YYSTYPE *sr, YYSTYPE *dst, char *dum) 
{
	anyResult *ar;

	if(dum) yyerror("parse error");
	if(!sr || !sr->text) return 0.0;
	parse_level++;
	ar = do_formula(0L, sr->text);
	yylval.a_data = 0L;	yylval.a_count = 0;
	switch(ar->type) {
	case ET_VALUE:
		dst->type = NUM;
		dst->val = ar->value;
		dst->text = 0L;
		break;
	case ET_TEXT:
		dst->type = STR;
		dst->val = 0.0;
		dst->text = PushString(ar->text);
		break;
	default:
		dst->type = NUM;
		dst->val = 0.0;
		dst->text = 0L;
		break;
		}
	parse_level--;
	return dst->val;
}

// more functions
static double sign(double v)
{
	if(v > 0.0) return 1.0;
	if(v < 0.0) return -1.0;
	return 0.0;
}

static long idum=0;
static double rand1(YYSTYPE *dst, YYSTYPE *src)
{
	if(!dst) return 0.0;
	dst->type = NUM;
	return(dst->val = ran2(&idum));
}

static double srand(double v)
{
	idum = (long)v;
	return v;
}

static double factorial(double v)
{
	return factrl((int)v);
}

static double _strlen(YYSTYPE *sr, YYSTYPE *dst, char *dum)
{
	if(dum) yyerror("parse error");
	if(!sr || !sr->text) return 0.0;
	return (double)strlen(sr->text);
}

#undef min
static double min(YYSTYPE *sr) 
{
	int i;

	if(!sr || !sr->a_count) return 0.0;
	if(sr->a_data && sr->a_data){
		for(i = 1, sr->val = sr->a_data[0]; i < sr->a_count; i++) 
			if(sr->a_data[i] < sr->val) sr->val = sr->a_data[i];
		}
	return sr->val;
}

#undef max
static double max(YYSTYPE *sr) 
{
	int i;

	if(!sr || !sr->a_count) return 0.0;
	if(sr->a_data){
		for(i = 1, sr->val = sr->a_data[0]; i < sr->a_count; i++) 
			if(sr->a_data[i] > sr->val) sr->val = sr->a_data[i];
		}
	return sr->val;
}

static double count(YYSTYPE *sr)
{
	if(!sr) return 0.0;
	if(sr->a_data) sr->val = (double)sr->a_count;
	else sr->val = 0.0;
	return sr->val;
}

static double sum(YYSTYPE *sr) 
{
	int i;

	if(!sr) return 0.0;
	if(sr->a_data){
		for(i = 0, sr->val = 0.0; i < sr->a_count; i++) sr->val += sr->a_data[i];
		}
	else sr->val = 0.0;
	return sr->val;
}

static double mean(YYSTYPE *sr) 
{
	if(!sr) return 0.0;
	if(sr->a_data && sr->a_count){
		sr->val = d_amean(sr->a_count, sr->a_data );
		}
	return sr->val;
}

static double kurt(YYSTYPE *sr) 
{
	if(!sr) return 0.0;
	if(sr->a_data && sr->a_count > 3){
		sr->val = d_kurt(sr->a_count, sr->a_data );
		}
	return sr->val;
}

static double skew(YYSTYPE *sr) 
{
	if(!sr) return 0.0;
	if(sr->a_data && sr->a_count > 2){
		sr->val = d_skew(sr->a_count, sr->a_data );
		}
	return sr->val;
}

static double gmean(YYSTYPE *sr) 
{
	int i;

	if(!sr) return 0.0;
	if(sr->a_data && sr->a_count){
		for(i = 0; i < sr->a_count; i++) if(sr->a_data[i] <= 0.0) {
			last_err_desc = "#VALUE";
			return sr->val = 0.0;
			}
		sr->val = d_gmean(sr->a_count, sr->a_data );
		}
	return sr->val;
}

static double hmean(YYSTYPE *sr) 
{
	int i;

	if(!sr) return 0.0;
	if(sr->a_data && sr->a_count){
		for(i = 0; i < sr->a_count; i++) if(sr->a_data[i] <= 0.0) {
			last_err_desc = "#VALUE";
			return sr->val = 0.0;
			}
		sr->val = d_hmean(sr->a_count, sr->a_data );
		}
	return sr->val;
}

static double quartile1(YYSTYPE *sr)
{
	if(!sr) return 0.0;
	if(sr->a_data && sr->a_count){
		d_quartile(sr->a_count, sr->a_data, &sr->val, 0L, 0L);
		}
	return sr->val;
}

static double quartile2(YYSTYPE *sr)
{
	if(!sr) return 0.0;
	if(sr->a_data && sr->a_count){
		d_quartile(sr->a_count, sr->a_data, 0L, &sr->val, 0L);
		}
	return sr->val;
}

static double quartile3(YYSTYPE *sr)
{
	if(!sr) return 0.0;
	if(sr->a_data && sr->a_count){
		d_quartile(sr->a_count, sr->a_data, 0L, 0L, &sr->val);
		}
	return sr->val;
}

static double variance(YYSTYPE *sr)
{
	if(!sr) return 0.0;
	sr->val = 0.0;
	if(sr->a_data && sr->a_count){
		sr->val = d_variance(sr->a_count, sr->a_data);
		}
	return sr->val;
}

static double stdev(YYSTYPE *sr)
{
	if(!sr) return 0.0;
	sr->val = 0.0;
	if(sr->a_data && sr->a_count){
		sr->val = sqrt(d_variance(sr->a_count, sr->a_data));
		}
	return sr->val;
}

static double sterr(YYSTYPE *sr)
{
	if(!sr) return 0.0;
	sr->val = 0.0;
	if(sr->a_data && sr->a_count){
		sr->val = sqrt(d_variance(sr->a_count, sr->a_data))/sqrt(sr->a_count);
		}
	return sr->val;
}

static double beta(YYSTYPE *sr)
{
	if(!sr) return 0.0;
	sr->val = 0.0;
        if(sr->a_data && sr->a_count == 2){
		sr->val = betaf(sr->a_data[0], sr->a_data[1]);
		}
	else yyargserr("Wrong number of arguments\nin call to  beta(u, v).");
	return sr->val;
}

static double _gammp(YYSTYPE *sr)
{
	if(!sr) return 0.0;
	sr->val = 0.0;
        if(sr->a_data && sr->a_count == 2){
		sr->val = gammp(sr->a_data[0], sr->a_data[1]);
		}
	else yyargserr("Wrong number of arguments\nin call to  gammp(a, x).");
	return sr->val;
}

static double _gammq(YYSTYPE *sr)
{
	if(!sr) return 0.0;
	sr->val = 0.0;
        if(sr->a_data && sr->a_count == 2){
		sr->val = gammq(sr->a_data[0], sr->a_data[1]);
		}
	else yyargserr("Wrong number of arguments\nin call to  gammq(a, x).");
	return sr->val;
}

static double _betai(YYSTYPE *sr)
{
	if(!sr) return 0.0;
	sr->val = 0.0;
        if(sr->a_data && sr->a_count == 3){
		if(sr->a_data[2] < 0.0 || sr->a_data[2] > 1.0) {
			last_err_desc = "#VALUE";
			return sr->val = 0.0;
			}
		sr->val = betai(sr->a_data[0], sr->a_data[1], sr->a_data[2]);
		}
	else yyargserr("Wrong number of arguments\nin call to  betai(a, b, x).");
	return sr->val;
}

static double _bincof(YYSTYPE *sr)
{
	if(!sr) return 0.0;
	sr->val = 0.0;
        if(sr->a_data && sr->a_count == 2){
		sr->val = bincof(sr->a_data[0], sr->a_data[1]);
		}
	else yyargserr("Wrong number of arguments\nin call to  bincof(n, k).");
	return sr->val;
}

static double binomdist(YYSTYPE *sr)
{
	if(!sr) return 0.0;
	sr->val = 0.0;
        if(sr->a_data && sr->a_count == 3){
		sr->val = binomdistf(sr->a_data[0], sr->a_data[1], sr->a_data[2]);
		}
	else yyargserr("Wrong number of arguments\nin call to  binomdist(s, n, p).");
	return sr->val;
}

static double binomfreq(YYSTYPE *sr)
{
	if(!sr) return 0.0;
	sr->val = 0.0;
        if(sr->a_data && sr->a_count == 3){
		sr->val = bincof(sr->a_data[1], sr->a_data[0]);
		sr->val *= pow(sr->a_data[2], sr->a_data[0]);
		sr->val *= pow(1.0 - sr->a_data[2], sr->a_data[1] - sr->a_data[0]);
		}
	else yyargserr("Wrong number of arguments\nin call to  binomfreq(s, n, p).");
	return sr->val;
}

static double normdist(YYSTYPE *sr)
{
	if(!sr) return 0.0;
	sr->val = 0.0;
	if(sr->a_data && sr->a_count == 3){
		sr->val = norm_dist(sr->a_data[0], sr->a_data[1], sr->a_data[2]);
		}
	else yyargserr("Wrong number of arguments\nin call to  normdist(x, mean, SD).");
	return sr->val;
}

static double norminv(YYSTYPE *sr)
{
	if(!sr) return 0.0;
	sr->val = 0.0;
        if(sr->a_data && sr->a_count == 3) {
		sr->val = distinv(norm_dist,sr->a_data[1], sr->a_data[2], sr->a_data[0], 2.0);
		}
	else yyargserr("Wrong number of arguments\nin call to  norminv(p, mean, SD).");
	return sr->val;
}

static double normfreq(YYSTYPE *sr)
{
	if(!sr) return 0.0;
	sr->val = 0.0;
	if(sr->a_data && sr->a_count == 3){
		sr->val = norm_freq(sr->a_data[0], sr->a_data[1], sr->a_data[2]);
		}
	else yyargserr("Wrong number of arguments\nin call to  normfreq(x, mean, SD).");
	return sr->val;
}

static double expdist(YYSTYPE *sr)
{
	if(!sr) return 0.0;
	sr->val = 0.0;
	if(sr->a_data && sr->a_count == 2){
		sr->val = exp_dist(sr->a_data[0], sr->a_data[1], 0.0);
		}
	else yyargserr("Wrong number of arguments\nin call to  expdist(x, l).");
	return sr->val;
}

static double expinv(YYSTYPE *sr)
{
	if(!sr) return 0.0;
	sr->val = 0.0;
        if(sr->a_data && sr->a_count == 2) {
		sr->val = exp_inv(sr->a_data[0], sr->a_data[1], 0.0);
		}
	else yyargserr("Wrong number of arguments\nin call to  expinv(p, l).");
	return sr->val;
}

static double expfreq(YYSTYPE *sr)
{
	if(!sr) return 0.0;
	sr->val = 0.0;
	if(sr->a_data && sr->a_count == 2){
		sr->val = exp_freq(sr->a_data[0], sr->a_data[1], 0.0);
		}
	else yyargserr("Wrong number of arguments\nin call to  expfreq(x, l).");
	return sr->val;
}

static double lognormdist(YYSTYPE *sr)
{
	if(!sr) return 0.0;
	sr->val = 0.0;
	if(sr->a_data && sr->a_count == 3){
		sr->val = lognorm_dist(sr->a_data[0], sr->a_data[1], sr->a_data[2]);
		}
	else yyargserr("Wrong number of arguments\nin call to  lognormdist(x, mean, SD).");
	return sr->val;
}

static double lognormfreq(YYSTYPE *sr)
{
	if(!sr) return 0.0;
	sr->val = 0.0;
	if(sr->a_data && sr->a_count == 3){
		sr->val = lognorm_freq(sr->a_data[0], sr->a_data[1], sr->a_data[2]);
		}
	else yyargserr("Wrong number of arguments\nin call to  lognormfreq(x, mean, SD).");
	return sr->val;
}

static double lognorminv(YYSTYPE *sr)
{
	if(!sr) return 0.0;
	sr->val = 0.0;
        if(sr->a_data && sr->a_count == 3) {
		sr->val = distinv(lognorm_dist,sr->a_data[1], sr->a_data[2], sr->a_data[0], 2.0);
		}
	else yyargserr("Wrong number of arguments\nin call to  lognorminv(p, mean, SD).");
	return sr->val;
}

static double chidist(YYSTYPE *sr)
{
	if(!sr) return 0.0;
	sr->val = 0.0;
	if(sr->a_data && sr->a_count == 2){
		sr->val = chi_dist(sr->a_data[0], sr->a_data[1], 1.0);
		}
	else yyargserr("Wrong number of arguments\nin call to  chidist(x, df).");
	return sr->val;
}

static double chiinv(YYSTYPE *sr)
{
	if(!sr) return 0.0;
	sr->val = 0.0;
        if(sr->a_data && sr->a_count == 2) {
		sr->val = distinv(chi_dist,sr->a_data[1], 1.0, sr->a_data[0], 2.0);
		}
	else yyargserr("Wrong number of arguments\nin call to  chiinv(p, df).");
	return sr->val;
}

static double tdist(YYSTYPE *sr)
{
	if(!sr) return 0.0;
	sr->val = 0.0;
	if(sr->a_data && sr->a_count == 2){
		sr->val = t_dist(sr->a_data[0], sr->a_data[1], 1.0);
		}
	else yyargserr("Wrong number of arguments\nin call to  tdist(x, df).");
	return sr->val;
}

static double tinv(YYSTYPE *sr)
{
	if(!sr) return 0.0;
	sr->val = 0.0;
        if(sr->a_data && sr->a_count == 2) {
		sr->val = distinv(t_dist,sr->a_data[1], 1.0, sr->a_data[0], 2.0);
		}
	else yyargserr("Wrong number of arguments\nin call to  tinv(p, df).");
	return sr->val;
}

static double poisdist(YYSTYPE *sr)
{
	if(!sr) return 0.0;
	sr->val = 0.0;
	if(sr->a_data && sr->a_count == 2){
		sr->val = pois_dist(sr->a_data[0], sr->a_data[1], 1.0);
		}
	else yyargserr("Wrong number of arguments\nin call to  poisdist(x, mean).");
	return sr->val;
}

static double poisfreq(YYSTYPE *sr)
{
	if(!sr) return 0.0;
	sr->val = 0.0;
	if(sr->a_data && sr->a_count == 2){
		sr->val = exp(log(sr->a_data[1])*sr->a_data[0] - sr->a_data[1] - gammln(1.0 + sr->a_data[0]));
		}
	else yyargserr("Wrong number of arguments\nin call to  poisfreq(x, mean).");
	return sr->val;
}

static double fdist(YYSTYPE *sr)
{
	if(!sr) return 0.0;
	sr->val = 0;
	if(sr->a_data && sr->a_count == 3){
		sr->val = f_dist(sr->a_data[0], sr->a_data[1], sr->a_data[2]);
		}
	else yyargserr("Wrong number of arguments\nin call to  fdist(x, df1, df2).");
	return sr->val;
}

static double finv(YYSTYPE *sr)
{
	if(!sr) return 0.0;
	sr->val = 0;
	if(sr->a_data && sr->a_count == 3){
		sr->val = distinv(f_dist,sr->a_data[1], sr->a_data[2], sr->a_data[0], 2.0);
		}
	else yyargserr("Wrong number of arguments\nin call to  finv(p, df1, df2).");
	return sr->val;
}

static double pearson(YYSTYPE *sr1, YYSTYPE *sr2, char *dest)
{
	if(!sr1 || !sr2) return 0.0;
	sr1->val = 0.0;
        if(sr1->a_data && sr1->a_count > 1 && sr2->a_data && sr1->a_count == sr2->a_count){
		sr1->val = sr2->val = d_pearson(sr1->a_data, sr2->a_data, sr1->a_count, dest, curr_data);
		}
	else yyargserr("Bad arguments in call to function\npearson(range1; range2 [;\"dest\"]).");
	return sr1->val;
}

static double spearman(YYSTYPE *sr1, YYSTYPE *sr2, char *dest)
{
	if(!sr1 || !sr2) return 0.0;
	sr1->val = 0.0;
        if(sr1->a_data && sr1->a_count > 1 && sr2->a_data && sr1->a_count == sr2->a_count){
		sr1->val = sr2->val = d_spearman(sr1->a_data, sr2->a_data, sr1->a_count, dest, curr_data);
		}
	else yyargserr("Bad arguments in call to function\nspearman(range1; range2 [;\"dest\"]).");
	return sr1->val;
}

static double kendall(YYSTYPE *sr1, YYSTYPE *sr2, char *dest)
{
	if(!sr1 || !sr2) return 0.0;
	sr1->val = 0.0;
        if(sr1->a_data && sr1->a_count > 1 && sr2->a_data && sr1->a_count == sr2->a_count){
		sr1->val = sr2->val = d_kendall(sr1->a_data, sr2->a_data, sr1->a_count, dest, curr_data);
		}
	else yyargserr("Bad arguments in call to function\nkendall(range1; range2 [;\"dest\"]).");
	return sr1->val;
}

static double regression(YYSTYPE *sr1, YYSTYPE *sr2, char *dest)
{
	if(!sr1 || !sr2) return 0.0;
	sr1->val = 0.0;
	if(!(dest)) yyargserr("No destination range in call to function\nregression(range1; range2; \"dest\").");
        if(sr1->a_data && sr1->a_count > 1 && sr2->a_data && sr1->a_count == sr2->a_count){
		sr1->val = sr2->val = d_regression(sr1->a_data, sr2->a_data, sr1->a_count, dest, curr_data);
		}
	else yyargserr("Bad arguments in call to function\nregression(range1; range2; \"dest\").");
	return sr1->val;
}

static double covar(YYSTYPE *sr1, YYSTYPE *sr2, char *dest)
{
	if(!sr1 || !sr2) return 0.0;
	sr1->val = 0.0;
        if(sr1->a_data && sr1->a_count > 1 && sr2->a_data && sr1->a_count == sr2->a_count){
		sr1->val = sr2->val = d_covar(sr1->a_data, sr2->a_data, sr1->a_count, dest, curr_data);
		}
	else yyargserr("Bad arguments in call to function\ncovar(range1; range2).");
	return sr1->val;
}

static double ttest(YYSTYPE *sr1, YYSTYPE *sr2, char *dest)
{
	if(!sr1 || !sr2) return 0.0;
	sr1->val = 0.0;
        if(sr1->a_data && sr1->a_count > 1 && sr2->a_data && sr2->a_count > 1){
		sr1->val = sr2->val = d_ttest(sr1->a_data, sr2->a_data, sr1->a_count, sr2->a_count, dest, curr_data);
		}
	else yyargserr("Bad arguments in call to function\nttest(array1; array2[;\"dest\"]).");
	return sr1->val;
}

static double ttest2(YYSTYPE *sr1, YYSTYPE *sr2, char *dest)
{
	if(!sr1 || !sr2) return 0.0;
	sr1->val = 0.0;
        if(sr1->a_data && sr1->a_count > 1 && sr2->a_data && sr2->a_count > 1){
		sr1->val = sr2->val = d_ttest2(sr1->a_data, sr2->a_data, sr1->a_count, dest, curr_data);
		}
	else yyargserr("Bad arguments in call to function\nttest2(range1; range2[;\"dest\"]).");
	return sr1->val;
}

static double utest(YYSTYPE *sr1, YYSTYPE *sr2, char *dest)
{
	if(!sr1 || !sr2) return 0.0;
	sr1->val = 0.0;
        if(sr1->a_data && sr1->a_count > 1 && sr2->a_data && sr2->a_count > 1){
		sr1->val = sr2->val = d_utest(sr1->a_data, sr2->a_data, sr1->a_count, sr2->a_count, dest, curr_data);
		}
	else yyargserr("Bad arguments in call to function\nutest2(array1; array2[;\"dest\"]).");
	return sr1->val;
}

static double ftest(YYSTYPE *sr1, YYSTYPE *sr2, char *dest)
{
	if(!sr1 || !sr2) return 0.0;
	sr1->val = 0.0;
        if(sr1->a_data && sr1->a_count > 1 && sr2->a_data && sr1->a_count > 1){
		sr1->val = sr2->val = d_ftest(sr1->a_data, sr2->a_data, sr1->a_count, sr2->a_count, dest, curr_data);
		}
	else yyargserr("Bad arguments in call to function\nftest(range1; range2[;\"dest\"]).");
	return sr1->val;
}

static double fill(YYSTYPE *sr, char *dest)
{
	AccRange *ar;
	int i, r, c;

	if(!sr || !sr->a_data || !sr->a_count || !dest || !dest[0]) return 0.0;
	if(ar = new AccRange(dest)) {
		for(i=0, ar->GetFirst(&c, &r); ar->GetNext(&c, &r) && i < sr->a_count; i++) {
			curr_data->SetValue(r, c, sr->a_data[i]);
			}
		delete ar;
		}
	return sr->val = i;
}

static void datestr(YYSTYPE *dst, YYSTYPE *src, char *fmt)
{
	dst->text = PushString(value_date(src->val, fmt));
}

static double dateval(YYSTYPE *sr, YYSTYPE *dst, char *fmt)
{
	if(!sr || !sr->text) return 0.0;
	if(fmt && fmt[0] && date_value(sr->text, fmt, &dst->val)) return dst->val;
	if(date_value(sr->text, 0L, &dst->val)) return dst->val;
	else return dst->val = 0.0;
}

static double leapyear(double year)
{
	int y = (int)year;

	return (double)((y % 4 == 0 && y % 100 != 0) || y % 400 == 0);
}

static void today(YYSTYPE *dst, YYSTYPE *src)
{
	if(src) yyerror("parse error");
	if(!dst) return;
	dst->val = floor(now_today());
	dst->type = DATE1;
}

static void now(YYSTYPE *dst, YYSTYPE *src)
{
	if(src) yyerror("parse error");
	if(!dst) return;
	dst->val = now_today();			dst->val -= floor(dst->val);
	dst->type = TIME1;
}

static double year(double dv)
{
	int res;

	split_date(dv, &res, 0L, 0L, 0L, 0L, 0L, 0L, 0L);
	return (double)res;
}

static double month(double dv)
{
	int res;

	split_date(dv, 0L, &res, 0L, 0L, 0L, 0L, 0L, 0L);
	return (double)res;
}

static double dom(double dv)
{
	int res;

	split_date(dv, 0L, 0L, &res, 0L, 0L, 0L, 0L, 0L);
	return (double)res;
}

static double dow(double dv)
{
	int res;

	split_date(dv, 0L, 0L, 0L, &res, 0L, 0L, 0L, 0L);
	return (double)res;
}

static double doy(double dv)
{
	int res;

	split_date(dv, 0L, 0L, 0L, 0L, &res, 0L, 0L, 0L);
	return (double)res;
}

static double hours(double dv)
{
	int res;

	split_date(dv, 0L, 0L, 0L, 0L, 0L, &res, 0L, 0L);
	return (double)res;
}

static double minutes(double dv)
{
	int res;

	split_date(dv, 0L, 0L, 0L, 0L, 0L, 0L, &res, 0L);
	return (double)res;
}

static double seconds(double dv)
{
	double res;

	split_date(dv, 0L, 0L, 0L, 0L, 0L, 0L, 0L, &res);
	if(res < 0.0005) res = 0.0;
	return res;
}

static void fdate(YYSTYPE *dst, YYSTYPE *src)
{
	if(!dst || !src || src->type == ARR || src->type == STR) {
		yyerror("parse error");	
		return;
		}
	dst->type = DATE1;		dst->val = src->val;
}

static void fdatetime(YYSTYPE *dst, YYSTYPE *src)
{
	if(!dst || !src || src->type == ARR || src->type == STR) {
		yyerror("parse error");	
		return;
		}
	dst->type = DATETIME1;		dst->val = src->val;
}

static void ftime(YYSTYPE *dst, YYSTYPE *src)
{
	if(!dst || !src || src->type == ARR || src->type == STR) {
		yyerror("parse error");	
		return;
		}
	dst->type = TIME1;		dst->val = src->val;
}

static void asort(YYSTYPE *dst, YYSTYPE *src)
{
	if(!dst || !src) return;
	dst->type = ARR;
	switch(src->a_count) {
	case 0:
		dst->a_data = PushArray((double*)malloc(sizeof(double)));
		dst->a_count = 1;	dst->a_data[0] = dst->val = src->val;
		break;
	case 1:
		dst->a_data = PushArray((double*)malloc(sizeof(double)));
		dst->a_count = 1;	dst->a_data[0] = dst->val = src->a_data[0];
		break;
	default:
		dst->a_data = PushArray((double*)memdup(src->a_data, src->a_count * sizeof(double), 0));
		dst->a_count = src->a_count;
		SortArray(dst->a_count, dst->a_data);
		}
}

static void _crank(YYSTYPE *dst, YYSTYPE *src)
{
	double tmp;

	if(!dst || !src) return;
	dst->type = ARR;
	switch(src->a_count) {
	case 0:	case 1:
		dst->a_data = PushArray((double*)malloc(sizeof(double)));
		dst->a_count = 1;	dst->a_data[0] = dst->val = 1;
		break;
	default:
		dst->a_data = PushArray((double*)memdup(src->a_data, src->a_count * sizeof(double), 0));
		dst->a_count = src->a_count;
		crank(dst->a_count, dst->a_data, &tmp);
		}
}

static void ltrim(YYSTYPE *dst, YYSTYPE *src)
{
	if(!src || !dst || !src->text) return;
	dst->text = PushString(str_ltrim(strdup(src->text)));
	dst->type = STR;	dst->val = 0.0;
}

static void rtrim(YYSTYPE *dst, YYSTYPE *src)
{
	if(!src || !dst || !src->text) return;
	dst->text = PushString(str_rtrim(strdup(src->text)));
	dst->type = STR;	dst->val = 0.0;
}

static void trim(YYSTYPE *dst, YYSTYPE *src)
{
	if(!src || !dst || !src->text) return;
	dst->text = PushString(str_trim(strdup(src->text)));
	dst->type = STR;	dst->val = 0.0;
}

static double rank(YYSTYPE *sr) 
{
	if(sr->a_count < 2 || !sr->a_data) return 0.0;
	return d_rank(sr->a_count-1, sr->a_data+1, sr->a_data[0]);
}

static double classes(double start, double step, YYSTYPE *src, YYSTYPE *dest)
{
	return d_classes(curr_data, start, step, src->a_data, src->a_count, dest->text);
}

// Store strings in a list
static char **str_list = 0L;
static int n_str = 0;

static char *PushString(char *text)
{
	if(text && text[0]) {
		if(str_list = (char**)realloc(str_list, sizeof(char*)*(n_str+1)))
			str_list[n_str] = strdup(text);
		return str_list[n_str++];
		}
	return 0L;
}

//Store arrays in a list
static double **arr_list = 0L;
static int n_arr = 0;

static double *PushArray(double *arr)
{
	if(arr) {
		if(arr_list = (double**)realloc(arr_list, sizeof(double*)*(n_arr+1)))
			arr_list[n_arr] = arr;
		return arr_list[n_arr++];
		}
	return 0L;
}

static double *ReallocArray(double *arr, int size)
{
	int i;

	if(arr && size) {
		for(i = 0; i < n_arr; i++) if(arr_list[i] == arr) {
			arr_list[i] = (double*)realloc(arr, size);
			return arr_list[i];
			}
		arr = (double*)realloc(arr, size);
		return PushArray(arr);
		}
	return 0L;
}


//The symbol table: a chain of `struct symrec'
static symrec *sym_table, *sym_tab_first;

//Rearrange function table with previously used functions in front
void ArrangeFunctions()
{
	symrec *ptr, *ptr1, *ptr2, *next;

	for(ptr = sym_table, ptr1 = ptr2 = 0L; (ptr); ) {
		next = ptr->next;
		if(ptr->name) {
			ptr->next = ptr1;
			ptr1 = ptr;
			}
		else {
			ptr->next = ptr2;
			ptr2 = ptr;
			}
		ptr = next;
		}
	for(sym_table = 0L, ptr = ptr2; (ptr); ){
		next = ptr->next;
		ptr->next = sym_table;
		sym_table = ptr;
		ptr = next;
		}
	for(ptr = ptr1; (ptr); ){
		next = ptr->next;
		ptr->next = sym_table;
		sym_table = ptr;
		ptr = next;
		}
	sym_tab_first = sym_table;
	bRecent = false;
}

// Put arithmetic functions and predifened variables in table
#define INIT_SYM(TYP,NAM,FNC) {TYP,NAM,(double(*)(double))&FNC} 
void InitArithFuncs(DataObj *d)
{
	struct fdef {
		int f_type;
		char *name;
		double (*fnct)(double);
		};
	fdef fncts[] = {
	INIT_SYM(FUNC4, "classes", classes),
	INIT_SYM(AFNCT, "rank", rank),			INIT_SYM(YYFNC, "ltrim", ltrim),
	INIT_SYM(YYFNC, "rtrim", rtrim),		INIT_SYM(YYFNC, "trim", trim),
	INIT_SYM(YYFNC, "asort", asort),		INIT_SYM(YYFNC, "crank", _crank),
	INIT_SYM(SRFUNC, "datestr", datestr),		INIT_SYM(SFNCT, "dateval", dateval),
	INIT_SYM(BFNCT, "leapyear", leapyear),		INIT_SYM(YYFNC, "today", today),
	INIT_SYM(YYFNC, "now", now),			INIT_SYM(FNCT, "year", year),
	INIT_SYM(FNCT, "month", month),			INIT_SYM(FNCT, "dom", dom),
	INIT_SYM(FNCT, "dow", dow),			INIT_SYM(FNCT, "doy", doy),
	INIT_SYM(FNCT, "hours", hours),			INIT_SYM(FNCT, "minutes", minutes),
	INIT_SYM(FNCT, "seconds", seconds),		INIT_SYM(YYFNC, "date", fdate),
	INIT_SYM(FNCT, "datetime", fdatetime),		INIT_SYM(YYFNC, "time", ftime),
	INIT_SYM(FUNC1, "fill", fill),			INIT_SYM(FUNC2, "pearson", pearson),
	INIT_SYM(FUNC2, "spearman", spearman),		INIT_SYM(FUNC2, "kendall", kendall),		
	INIT_SYM(FUNC2, "correl", pearson),		INIT_SYM(FUNC2, "regression", regression),
	INIT_SYM(FUNC2, "covar", covar),
	INIT_SYM(FUNC3, "utest", utest),		INIT_SYM(FUNC2, "ttest2", ttest2),
	INIT_SYM(FUNC3, "ttest", ttest),		INIT_SYM(FUNC3, "ftest", ftest),
	INIT_SYM(AFNCT, "variance", variance),		INIT_SYM(AFNCT, "stdev", stdev),
	INIT_SYM(AFNCT, "sterr", sterr),		INIT_SYM(AFNCT, "min", min),
	INIT_SYM(AFNCT, "max", max),			INIT_SYM(AFNCT, "count", count),
	INIT_SYM(AFNCT, "sum", sum),			INIT_SYM(AFNCT, "mean", mean),
	INIT_SYM(AFNCT, "kurt", kurt),			INIT_SYM(AFNCT, "skew", skew),
	INIT_SYM(AFNCT, "median", quartile2),		INIT_SYM(AFNCT, "quartile1", quartile1),
	INIT_SYM(AFNCT, "quartile2",quartile2),		INIT_SYM(AFNCT, "quartile3", quartile3),
	INIT_SYM(AFNCT, "gmean", gmean),		INIT_SYM(AFNCT, "hmean", hmean),
	INIT_SYM(AFNCT, "tdist", tdist),		INIT_SYM(AFNCT, "tinv", tinv),
	INIT_SYM(AFNCT, "poisdist", poisdist),		INIT_SYM(AFNCT, "poisfreq", poisfreq),		
	INIT_SYM(AFNCT, "expdist", expdist),		INIT_SYM(AFNCT, "expfreq", expfreq),		
	INIT_SYM(AFNCT, "expinv", expinv),		INIT_SYM(AFNCT, "fdist", fdist),
	INIT_SYM(AFNCT, "finv", finv),			INIT_SYM(AFNCT, "gammp", _gammp),
	INIT_SYM(AFNCT, "gammq", _gammq),		INIT_SYM(AFNCT, "beta", beta),
	INIT_SYM(AFNCT, "betai", _betai),		INIT_SYM(AFNCT, "bincof", _bincof),
	INIT_SYM(AFNCT, "binomdist",binomdist),
	INIT_SYM(AFNCT, "binomfreq",binomfreq),
	INIT_SYM(AFNCT, "normdist", normdist),
	INIT_SYM(AFNCT, "norminv", norminv),		INIT_SYM(AFNCT, "normfreq", normfreq),
	INIT_SYM(AFNCT, "lognormdist", lognormdist),	INIT_SYM(AFNCT, "lognormfreq", lognormfreq),
	INIT_SYM(AFNCT, "lognorminv",lognorminv),	INIT_SYM(AFNCT, "chidist", chidist),
	INIT_SYM(AFNCT, "chiinv", chiinv),		INIT_SYM(SFNCT, "strlen", _strlen),
	INIT_SYM(SFNCT, "eval", eval),			INIT_SYM(FNCT, "erf", errf),
	INIT_SYM(FNCT, "erfc", errfc),			INIT_SYM(FNCT, "sign", sign),
	INIT_SYM(FNCT, "gammaln", gammln),		INIT_SYM(FNCT, "factorial", factorial),
	INIT_SYM(YYFNC, "rand", rand1),			INIT_SYM(FNCT, "srand", srand),
	INIT_SYM(FNCT, "floor", floor),			INIT_SYM(FNCT, "abs", fabs),
	INIT_SYM(FNCT, "asin", asin),			INIT_SYM(FNCT, "acos", acos),
	INIT_SYM(FNCT, "atan", atan),			INIT_SYM(FNCT, "sinh", sinh),
	INIT_SYM(FNCT, "cosh", cosh),			INIT_SYM(FNCT, "tanh", tanh),
	INIT_SYM(FNCT, "sin", sin),			INIT_SYM(FNCT, "cos", cos),
	INIT_SYM(FNCT, "tan", tan),			INIT_SYM(FNCT, "log10", log10),
	INIT_SYM(FNCT, "ln", log),			INIT_SYM(FNCT, "log", log),
	INIT_SYM(FNCT, "exp", exp),			INIT_SYM(FNCT, "sqrt", sqrt),
	INIT_SYM(0, 0L, nop)};
	int i;
	symrec *ptr, *next;

	if(d) curr_data = d;
	if(sym_table) {
		for (ptr = sym_table; ptr != (symrec *) 0;){
			if(ptr) {
				next = ptr->next;
				delete (ptr);
				}
			ptr = next;
			}
		sym_table = sym_tab_first = (symrec *) 0;
		}
	for (i = 0; fncts[i].name; i++) {
		ptr = putsym (HashValue((unsigned char*) fncts[i].name), Hash2((unsigned char*) fncts[i].name), fncts[i].f_type);
		ptr->fnctptr = (double (*)(...))fncts[i].fnct;
		}
	ptr = putsym(HashValue((unsigned char*)"zdiv"), Hash2((unsigned char*)"zdiv"), VAR);	ptr->SetValue(1.0);
	sym_tab_first = sym_table;
}
#undef INIT_SYM

static void init_table (void)
{
	str_list = 0L;		n_str = 0;
	arr_list = 0L;		n_arr = 0;
	push_syntax();
}

static void clear_table()
{
	int i;

	if(str_list) {
		for(i = 0; i < n_str; i++) if(str_list[i]) free(str_list[i]);
		free(str_list);		str_list = 0L;		n_str = 0;
		}
	if(arr_list) {
		for(i = 0; i < n_arr; i++) if(arr_list[i]) free(arr_list[i]);
		free(arr_list);		arr_list = 0L;		n_arr = 0;
		}
	pop_syntax();
}

static symrec *
putsym (unsigned int h_name, unsigned int h2_name, int sym_type)
{
	sym_table = new symrec(h_name, h2_name, sym_type, sym_table);
	return sym_table;
}

static symrec *
getsym (unsigned int h_name, unsigned int h2_name, char *sym_name)
{
	symrec *ptr;

	if(!h_name) return 0;
	for (ptr = sym_table; ptr != (symrec *) 0; ptr = (symrec *)ptr->next) {
		if (ptr->h_name == h_name && ptr->h2_name == h2_name){
			if(sym_name && !ptr->name) {
				ptr->SetName(sym_name);
				bRecent = true;
				}
			return ptr;
			}
		//predefined variables rarely end on a digit
		else if(ptr == sym_tab_first) {
			if(sym_name && isdigit(sym_name[strlen(sym_name)-1]) && strlen(sym_name) < 5) return 0;
			}
		}
	return 0;
}

static int
push(YYSTYPE *res, YYSTYPE *val)
{
	if(val->a_data) {
		if(!(res->a_data)) {
			if(!(val->a_data=ReallocArray(val->a_data, (val->a_count+2)*sizeof(double))))return 0;
			val->a_data[val->a_count++] = res->val;
			res->a_data = val->a_data;		res->a_count = val->a_count;
			val->a_data = 0L;			val->a_count = 0;
			val->val = res->val;			return 1;
			}
		else {
			if(!(res->a_data=ReallocArray(res->a_data, (val->a_count+res->a_count)*sizeof(double))))return 0;
			memcpy(&res->a_data[res->a_count], val->a_data, val->a_count*sizeof(double));
			res->a_count += val->a_count;
			val->a_data = 0L;			val->a_count = 0;
			return 1;
			}
		}
	if(!(res->a_data )){
		if(!(res->a_data =  PushArray((double*)malloc(2*sizeof(double)))))return 0;
		res->a_data[0] = res->val;			res->a_data[1] = val->val;
		res->a_count = 2;
		return 1;
		}
	else {
		if(!(res->a_data = ReallocArray(res->a_data, (res->a_count+2)*sizeof(double))))return 0; 
		res->a_data[res->a_count] = val->val;		res->a_count++;
		return 1;
		}
	return 0;
}

static int
range_array(YYSTYPE * res, char *range)
{
	AccRange *r;
	int row, col;
	anyResult ares;

	if(!range || !range[0] || !(r = new AccRange(range))) return 0;
	if(!r->GetFirst(&col, &row) || !(res->a_data =  PushArray((double*)malloc(r->CountItems() * sizeof(double))))) {
		delete(r);
		return 0;
		}
	parse_level++;
	for(res->a_count = 0; r->GetNext(&col, &row); ) {
		if(curr_data->GetResult(&ares, row, col, parse_level > MAX_PARSE)) {
			switch(ares.type) {
			case ET_VALUE:	case ET_TIME:	case ET_DATE:	case ET_DATETIME:	case ET_BOOL:
				res->a_data[res->a_count++] = ares.value;
				break;
				}
			}
		}
	parse_level--;
	delete(r);
	return 1;
}

static int
range_array2(YYSTYPE *res1, YYSTYPE *res2)
{
	AccRange *r1, *r2;
	int row1, col1, row2, col2;
	anyResult ares1, ares2;
	char *range1, *range2;

	range1 = res1->text;	range2 = res2->text;
	if(!range1 || !range1[0] || !range2 || !range2[0] || !(r1 = new AccRange(range1)) 
		|| !(r2 = new AccRange(range2))) return 0;
	if(!r1->GetFirst(&col1, &row1) || !(res1->a_data =  PushArray((double*)malloc(r1->CountItems() * sizeof(double))))) {
		delete(r1);	delete(r2);
		return 0;
		}
	if(!r2->GetFirst(&col2, &row2) || !(res2->a_data =  PushArray((double*)malloc(r2->CountItems() * sizeof(double))))) {
		delete(r1);	delete(r2);
		return 0;
		}
	parse_level++;
	for(res1->a_count = res2->a_count = 0; r1->GetNext(&col1, &row1) && r2->GetNext(&col2, &row2); ) {
		if(curr_data->GetResult(&ares1, row1, col1, parse_level > MAX_PARSE) 
			&& curr_data->GetResult(&ares2, row2, col2, parse_level > MAX_PARSE)
			&& (ares1.type==ET_VALUE || ares1.type==ET_TIME || ares1.type==ET_DATE || ares1.type==ET_DATETIME || ares1.type==ET_BOOL)
			&& (ares2.type==ET_VALUE || ares2.type==ET_TIME || ares2.type==ET_DATE || ares2.type==ET_DATETIME || ares2.type==ET_BOOL)){
			res1->a_data[res1->a_count++] = ares1.value;
			res2->a_data[res2->a_count++] = ares2.value;
			}
		}
	parse_level--;
	delete(r1);	delete(r2);
	return 1;
}

static YYSTYPE *proc_clause(YYSTYPE *res)
{
	int i, n, o_pos;
	char *o_cmd;
	double *n_data;

	if(!(syntax_level) || !syntax_level->cl1 || syntax_level->cl2 <= syntax_level->cl1) return res;
	if(!res->text) return res;
	if(!res->a_data && (res->a_data = PushArray((double*)malloc(sizeof(double))))) {
		res->a_data[0] = res->type == VAR && res->tptr ? res->tptr->GetValue() : res->val;
		res->a_count = 1;
		}
	else if(!res->a_data) return res;
	if(!(n_data = PushArray((double*)malloc(res->a_count * sizeof(double))))) return res;
	o_pos = buff_pos;	o_cmd = buffer;
	for(i = n = 0; i < res->a_count; i++) {
		buffer = res->text;	buff_pos = 0;
		if(!syntax_level) break;
		syntax_level->clval = res->a_data[i];
		yyparse();
		if(line_res.type == ET_VALUE && line_res.value != 0.0) n_data[n++] = res->a_data[i];
		}
	res->a_data = n_data;		res->a_count = n;
	res->text=0L;
	syntax_level->cl1 = syntax_level->cl2 = 0;
	buffer = o_cmd;	buff_pos = o_pos;
	return res;
}

static void exec_clause(YYSTYPE *res)
{
	int i, j;
	char *cmd;

	if((!res->a_data || res->a_count <2) && res->text && res->text[0]) range_array(res, res->text);
	if(!res->a_data) {
		if(res->a_data = PushArray((double*)malloc(2*sizeof(double)))) {
			res->a_data[0] = res->val;	res->a_count = 1;
			InfoBox("fixed data");
			}
		}
	if(!(syntax_level) || !syntax_level->cl1 || syntax_level->cl2 <= syntax_level->cl1) return;
	if(!(cmd = (char*)malloc(syntax_level->cl2 - syntax_level->cl1 +2)))return;
	while(buffer[syntax_level->cl1] <= ' ' && syntax_level->cl1 < syntax_level->cl2) syntax_level->cl1++;
	for(j = 0, i = syntax_level->cl1; i< syntax_level->cl2; i++) {
		cmd[j++] = buffer[i];
		}
	cmd[j++] = ';';		cmd[j++] = 0;
	res->text = PushString(cmd);
	free(cmd);
}

struct parse_info  {
	char *buffer;
	int buff_pos;
	DataObj *curr_data;
	symrec *sym_table;
	YYSTYPE yylval;
	struct parse_info *next;
	char **str_list;
	double **arr_list;
	char *last_err_desc;
	int n_str, n_arr, yychar, yynerrs;
};
static parse_info *parse_stack = 0L;

static void push_parser()
{
	parse_info *ptr;

	if(!sym_table) InitArithFuncs(0L);
	else if(!parse_level && bRecent) ArrangeFunctions();
	ptr = (parse_info *) malloc(sizeof(parse_info));
	ptr->buffer = buffer;			ptr->buff_pos = buff_pos;
	ptr->curr_data = curr_data;		ptr->last_err_desc = last_err_desc;
	ptr->sym_table = sym_table;		sym_table = sym_tab_first;
	memcpy(&ptr->yylval, &yylval, sizeof(YYSTYPE));
	ptr->next = parse_stack;
	ptr->str_list = str_list;		str_list = 0L;
	ptr->n_str = n_str;			n_str = 0;
	ptr->arr_list = arr_list;		arr_list = 0L;
	ptr->n_arr = n_arr;			n_arr = 0;
	ptr->yychar = yychar;			ptr->yynerrs = yynerrs;
	parse_stack = ptr;			last_err_desc = 0L;
	parse_level++;				//reenter ?
	push_syntax();				syntax_level->last_tok = 0;
}

static void pop_parser()
{
	parse_info *ptr;
	symrec *n;

	if(ptr = parse_stack) {
		while(sym_table  && sym_table != sym_tab_first) {
			n = sym_table->next;
			delete(sym_table);
			sym_table = n;
			}
		if(sym_table) sym_table = ptr->sym_table;
		parse_stack = ptr->next;
		buffer = ptr->buffer;		buff_pos = ptr->buff_pos;
		curr_data = ptr->curr_data;	last_err_desc = ptr->last_err_desc;
		memcpy(&yylval, &ptr->yylval, sizeof(YYSTYPE));
		str_list = ptr->str_list;	n_str = ptr->n_str;
		arr_list = ptr->arr_list;	n_arr = ptr->n_arr;
		yychar = ptr->yychar;		yynerrs = ptr->yynerrs;
		free(ptr);
		parse_level--;
		}
	pop_syntax();
}

static int is_ttoken(unsigned int h_nam, unsigned int h2_nam)
{
	switch(h_nam) {
	case 69:
		if(h2_nam == 101) return E;
		break;
	case 393:
		if(h2_nam == 47081) return PI;
		break;
	case 28381:
		if((h2_nam & 0x7fffffff) == 0x7c2706ed) {
			if(syntax_level) syntax_level->cl1 = buff_pos;
			return CLAUSE;
			}
		break;
	case 20:
		if(h2_nam == 5220) return CLVAL;
		break;
	case 362:
		if(h2_nam == 42878) return IF;
		break;
	case 1457:
		if(h2_nam == 18357885) return DIM;
		break;
	case 6033:
		if((h2_nam & 0x7fffffff) == 0x6371377d) return ELSE;
		break;
	case 7097:
		if((h2_nam & 0x7fffffff) == 0x550a2d65) return BTRUE;
		break;
	case 23697:
		if((h2_nam & 0x7fffffff) == 0x155f977d) return BFALSE;
		break;
		}
	return 0;
}

static symrec *curr_sym;
static int yylex (void)
{
	int i, c, tok;
	unsigned int h_nam, h2_nam;
	char tmp_txt[80];
	symrec *s;

	while((c = buffer[buff_pos++]) == ' ' || c == '\t');	//get first nonwhite char
	if(!c) return 0;
	//test for block statement
	if(c == '{') {
		for(i= 0; i < 79 && ((tok = buffer[buff_pos]) && (tok != '}')); buff_pos++) {
			tmp_txt[i++] = (char)tok;
			}
		if(buffer[buff_pos] == '}')buff_pos++;
		tmp_txt[i] = 0;
		yylval.text = PushString(tmp_txt);
		return yylval.type = BLOCK;
		}
	//test for '..' operator
	if(c == '.' && buffer[buff_pos] == '.') {
		buff_pos++;
		return yylval.type = SER;
		}
	//test for number
	if(c == '.' || isdigit(c)) {
		for(buff_pos--, i = 0; i < 79 && ((c = buffer[buff_pos]) == '.' || isdigit(c)); buff_pos++) {
			tmp_txt[i++] = (char)c;
			if(i && buffer[buff_pos+1] == 'e' && (buffer[buff_pos+2] == '-' || buffer[buff_pos+2] == '+')){
				tmp_txt[i++] = buffer[++buff_pos];
				tmp_txt[i++] = buffer[++buff_pos];
				}
			if(i && buffer[buff_pos+1] == '.' && buffer[buff_pos+2]  == '.') {	//operator '..'
				buff_pos++;
				break;
				}
			}
		tmp_txt[i] = 0;
		sscanf(tmp_txt, "%lf", &yylval.val);
		return yylval.type = NUM;
		}
	//test for name or stringtoken
	if(isalpha(c) || c=='$') {
 		for(buff_pos--, i = 0; i < 79 && ((c = buffer[buff_pos]) && (isalnum(c) || c == '$')); buff_pos++) {
			tmp_txt[i++] = (char)c; 
			}
		tmp_txt[i] = 0;
		h_nam = HashValue((unsigned char*)tmp_txt);
		h2_nam = Hash2((unsigned char*)tmp_txt);
		if(h_nam == 1550 && h2_nam == 18852086) {	//'inf' = huge value
			yylval.val = HUGE_VAL;
			return yylval.type = NUM;
			}
		if(tok = is_ttoken(h_nam, h2_nam)) return tok;
		if(!(s = getsym(h_nam, h2_nam, tmp_txt))){
			s = putsym(h_nam, h2_nam, VAR);
			s->SetName(tmp_txt);
			}
		
		curr_sym = yylval.tptr = s;	return s->type;
		}
	//test for string
	if(c == '"' || c == '\'') {
		for(i= 0; i < 79 && ((tok = buffer[buff_pos]) && (tok != c)); buff_pos++) {
			tmp_txt[i++] = (char)tok;
			}
		if(buffer[buff_pos] == c)buff_pos++;
		tmp_txt[i] = 0;
		yylval.text = PushString(tmp_txt);
		return yylval.type = STR;
		}
	tok = 0;
	switch(c) {
	case '=':
		if(buffer[buff_pos] == '=') tok = EQ;
		break;
	case '!':
		if(buffer[buff_pos] == '=') tok = NE;
		break;
	case '>':
		if(buffer[buff_pos] == '=') tok = GE;
		else return GT;
		break;
	case '<':
		if(buffer[buff_pos] == '=') tok = LE;
		else if(buffer[buff_pos] == '>') tok = NE;
		else return LT;
		break;
	case '&':
		if(buffer[buff_pos] == '&') tok = AND;
		break;
	case '|':
		if(buffer[buff_pos] == '|') tok = OR;
		break;
	case ')':
		if(syntax_level) {
			if(syntax_level->cl1 && syntax_level->next) {
				syntax_level->next->cl1 = syntax_level->cl1;
				syntax_level->next->cl2 = buff_pos-1;
				}
			}
		pop_syntax();
		break;
	case '(':
		push_syntax();
	case '?':
		if(syntax_level) syntax_level->last_tok = c;
		break;
	case ':':
		if(syntax_level) {
			if(syntax_level->last_tok == '(') return COLR;
			else if(syntax_level->last_tok == '?') return COLC;
			}
		break;
	case ';':
		if(syntax_level) {
			if(syntax_level->last_tok == '(') return PSEP;
			}
		break;
	case '+':
		if(buffer[buff_pos] == '+') tok = INC;
		break;
	case '-':
		if(buffer[buff_pos] == '-') tok = DEC;
		break;
		}
	if(tok) {
		buff_pos++;		return tok;
		}
	//Any other character is a token by itself
	return c;
}

static unsigned int hn_x = HashValue((unsigned char *)"x");
static unsigned int hn_y = HashValue((unsigned char *)"y");
static unsigned int hn_z = HashValue((unsigned char *)"z");
static unsigned int h2_x = Hash2((unsigned char *)"x");
static unsigned int h2_y = Hash2((unsigned char *)"y");
static unsigned int h2_z = Hash2((unsigned char *)"z");

bool do_xyfunc(DataObj *d, double x1, double x2, double step, char *expr, lfPOINT **pts, long *npts, char *param)
{
	double x, y;
	symrec *sx, *sy;
	lfPOINT *new_points;
	long npoints = 0;
	int length, res_mode = 0;

	if(x1 < x2) step = fabs(step);
	else step = -fabs(step);
	if(!(new_points = (lfPOINT*)calloc((iround(fabs(x2-x1)/fabs(step))+2), sizeof(lfPOINT))))
		return false;
	if(d) curr_data = d;
	push_parser();
	init_table();
	if(param) {
		length = strlen(param);
		if(!(buffer = (char*)malloc(length+2))){
			pop_parser();
			return false;
			}
		strcpy(buffer, param);	buffer[length++] = ';';
		buffer[length] = 0;	buff_pos = 0;
		do {
			yyparse();
			}while(buff_pos < length);
		free(buffer);		buffer = 0L;
		}		
	length = strlen(expr);
	buffer = expr;		sx = putsym(hn_x, h2_x, VAR);
	for(x = x1; step > 0.0 ? x <= x2 : x >= x2; x += step) {
		if(sx){
			sx->SetValue(x);	buff_pos = 0;
			do {
				yyparse();
				}while(buff_pos < length);
			switch (res_mode) {
			case 1:
				y = sy->GetValue();	break;
			case 2:
				y = line_res.value;	break;
			default:
				if(sy = getsym(hn_y, h2_y)) {
					y = sy->GetValue();	res_mode = 1;
					}
				else {
					y = line_res.value;	res_mode = 2;
					}
				break;
				}
			new_points[npoints].fx = (getsym(hn_x, h2_x))->GetValue();
			new_points[npoints++].fy = y;
			}
		}
	*pts = new_points;	*npts = npoints;
	clear_table();
	pop_parser();
	if(curr_data) {
		curr_data->Command(CMD_CLEAR_ERROR, 0L, 0L);
		curr_data->Command(CMD_REDRAW, 0L, 0L);
		}
	return true;
}

bool do_func3D(DataObj *d, double x1, double x2, double xstep, double z1, double z2, double zstep, 
	char *expr, char *param)
{
	int length, nr, nc, r, c, res_mode=0;
	symrec *sx, *sz, *sy;
	double x, y, z;

	if(!d || x2 <= x1 || z2 <= z1 || xstep <= 0.0 || zstep <= 0.0) return false;
	push_parser();
	init_table();
	if(param) {
		length = strlen(param);
		if(!(buffer = (char*)malloc(length+2))){
			pop_parser();
			return false;
			}
		strcpy(buffer, param);	buffer[length++] = ';';
		buffer[length] = 0;	buff_pos = 0;
		do {
			yyparse();
			}while(buff_pos < length);
		free(buffer);		buffer = 0L;
		}		
	length = strlen(expr);		buffer = expr;
	sx = putsym(hn_x, h2_x, VAR);	sz = putsym(hn_z, h2_z, VAR);
	nr = iround((z2-z1)/zstep)+1;	nc = iround((x2-x1)/xstep)+1;
	d->Init(nr, nc);
	for(r = 0, x = x1; r < nr; r++, x += xstep) {
		for(c = 0, z = z1; c < nc; c++, z+= zstep) {
			sx->SetValue(x);	sz->SetValue(z);	buff_pos = 0;
			do {
				yyparse();
				}while(buff_pos < length);
			switch (res_mode) {
			case 1:
				y = sy->GetValue();	break;
			case 2:
				y = line_res.value;	break;
			default:
				if(sy = getsym(hn_y, h2_y)) {
					y = sy->GetValue();	res_mode = 1;
					}
				else {
					y = line_res.value;	res_mode = 2;
					}
				break;
				}
			d->SetValue(r, c, y);
			}
		} 
	clear_table();
	pop_parser();
	return true;
}

anyResult *do_formula(DataObj *d, char *expr)
{
	int length;
	static anyResult ret, *pret = 0L;

	if(d) curr_data = d;
	ret.type = ET_ERROR;		ret.text = 0L;
	if(!expr || !expr[0]) {
		if(!sym_table) InitArithFuncs(0L);
		return &ret;
		}
	push_parser();		//make code reentrant
	init_table();		length = strlen(expr);
	if(!(buffer = (char*)malloc(length+2))){
		pop_parser();
		return &ret;
		}
	strcpy(buffer, expr);	if(buffer[length-1] != ';') buffer[length++] = ';';
	buffer[length] = 0;	buff_pos = 0;
	do {
		yyparse();
		}while(buff_pos < length);
	ret.type = ET_ERROR;		ret.text = 0L;
	if(curr_data && last_error) {
		if(!(strcmp(last_error, "parse error"))) curr_data->Command(CMD_ERROR, 0L, 0L);
		if(last_err_desc) pret = &line_res;
		else pret = &ret;
		}
	else pret = &line_res;
	last_error = last_err_desc = 0L;
	free(buffer);		buffer = 0L;
	clear_table();
	pop_parser();
	return pret;
}

bool MoveFormula(DataObj *d, char *of, char *nf, int dx, int dy, int r0, int c0)
{
	int length, tok, pos, i;
	char *res, desc1[2], desc2[2];

	if(d) curr_data = d;
	if(!curr_data || !of || !nf) return false;
	push_parser();		//make code reentrant
	init_table();		length = strlen(of);
	if(!(buffer = (char*)malloc(length+2))){
		pop_parser();
		return false;
		}
	strcpy(buffer, of);	buffer[length++] = ';';
	buffer[length] = 0;	buff_pos = pos = 0;
	res = (char *)calloc(length*2+10, sizeof(char));
	do {
		tok = yylex ();
		if(tok && tok < 256) {
			if(res[pos-1] == ' ') pos--;
			res[pos++] = (char)tok;
			}
		else switch(tok) {
			case NUM:
				pos += sprintf(res+pos, "%g", yylval.val);
				break;
			case FNCT:	case FUNC1:	case FUNC2:	case FUNC3:	case AFNCT:
			case SFNCT:	case SRFUNC:	case BFNCT:	case YYFNC:	case FUNC4:
				pos += sprintf(res+pos, "%s", curr_sym->name);
				break;
			case COLR:			case COLC:
				pos += sprintf(res+pos, ":");
				break;
			case PSEP:
				pos += sprintf(res+pos, ";");
				break;
			case CLVAL:
				pos += sprintf(res+pos, "$$");
				break;
			case CLAUSE:
				pos += sprintf(res+pos, " where ");
				break;
			case VAR:
				curr_sym->InitSS();
				if(curr_sym->col >= 0 && curr_sym->row >= 0) {
					desc1[0] = desc1[1] = desc2[0] = desc2[1] = 0;
					for(i=strlen(curr_sym->name)-1; i>0 && isdigit(curr_sym->name[i]); i--);
					if(curr_sym->name[0] == '$') desc1[0] = '$';
					if(curr_sym->name[i] == '$') desc2[0] = '$';
					pos += sprintf(res+pos, "%s%s%s%d", desc1, 
						Int2ColLabel(desc1[0] || curr_sym->col < c0 ? curr_sym->col : curr_sym->col+dx >=0 ?
						curr_sym->col+dx > c0 ? curr_sym->col+dx : c0 : 0, false),
						desc2, desc2[0] || curr_sym->row < r0 ? curr_sym->row+1 : curr_sym->row + dy >= 0 ? 
						curr_sym->row+dy > r0 ? curr_sym->row+1+dy : r0 : 1);
					}
				else pos += sprintf(res+pos, "%s ", curr_sym->name);
				break;
			case STR:
				pos += sprintf(res+pos, "\"%s\"", yylval.text && yylval.text[0] ? yylval.text : "");
				break;
			case SER:
				pos += sprintf(res+pos, "..");
				break;
			case INC:
				pos += sprintf(res+pos, "++");
				break;
			case DEC:
				pos += sprintf(res+pos, "--");
				break;
			case PI:
				pos += sprintf(res+pos, "pi");
				break;
			case E:
				pos += sprintf(res+pos, "e");
				break;
			case BTRUE:
				pos += sprintf(res+pos, "true");
				break;
			case BFALSE:
				pos += sprintf(res+pos, "false");
				break;
			case AND:
				pos += sprintf(res+pos, " && ");
				break;
			case OR:
				pos += sprintf(res+pos, " || ");
				break;
			case EQ:
				pos += sprintf(res+pos, " == ");
				break;
			case NE:
				pos += sprintf(res+pos, " != ");
				break;
			case GT:
				pos += sprintf(res+pos, ">");
				break;
			case GE:
				pos += sprintf(res+pos, ">=");
				break;
			case LT:
				pos += sprintf(res+pos, "<");
				break;
			case LE:
				pos += sprintf(res+pos, "<=");
				break;
			case IF: 
				pos += sprintf(res+pos, "if");
				break;
			case ELSE: 
				pos += sprintf(res+pos, "else");
				break;
			case BLOCK:
				pos += sprintf(res+pos, "{%s}", yylval.text && yylval.text[0] ? yylval.text : "");
				break;
			}
		}while(buff_pos < length);
	while((res[pos-1] == ';' || res[pos-1] == ' ') && pos > 0) { res[pos-1] = 0; pos--;} 
	strcpy(nf, res);	free(res);
	free(buffer);		buffer = 0L;
	clear_table();
	pop_parser();
	return true;
}

static char *txt_formula;	//function to fit
static double **parval;		//pointers to parameter values
static void fcurve(double x, double z, double **a, double *y, double dyda[], int ma)
{
	int i, length;
	double tmp, y1, y2;
	symrec *symx, *symz, *sy=0L;

	if(!(symx = getsym(hn_x, h2_x))) symx = putsym(hn_x, h2_x, VAR);
	if(!(symz = getsym(hn_z, h2_z))) symz = putsym(hn_z, h2_z, VAR);
	//swap parameters to requested set
	if(a != parval) for(i = 0; i < ma; i++) {
		tmp = *parval[i];	*parval[i]  = *a[i];	*a[i] = tmp;
		}
	//calc result
	symx->SetValue(x);	symz->SetValue(z);	
	buffer = txt_formula;
	buff_pos = 0;		length = strlen(txt_formula);
	do {	yyparse();	}while(buff_pos < length);
	if(sy = getsym(hn_y, h2_y)) *y = sy->GetValue();
	else *y = line_res.value;
	if(*y == HUGE_VAL || *y == -HUGE_VAL) {
		for(i = 0, *y = 0.0; i < ma; dyda[i++] = 0.0);
		return;
		}
	//partial derivatives for each parameter by numerical differentiation
	for(i = 0; i < ma; i++) {
		if(*parval[i] != 0.0) {
			tmp = *parval[i];
			*parval[i] = tmp*.995;
			buff_pos = 0;
			do {	yyparse();	}while(buff_pos < length);
			y1 = sy ? sy->GetValue() : line_res.value;
			*parval[i] = tmp*1.005;
			buff_pos = 0;
			do {	yyparse();	}while(buff_pos < length);
			y2 = sy ? sy->GetValue() : line_res.value;
			*parval[i] = tmp;
			dyda[i] = (y2-y1)*100.0/tmp;
			}
		else dyda[i] = 0.0;
		}
	//swap parameters back to original
	if(a != parval) for(i = 0; i < ma; i++) {
		tmp = *parval[i];	*parval[i]  = *a[i];	*a[i] = tmp;
		}
}

int do_fitfunc(DataObj *d, char *rx, char *ry, char *rz, char **par, char *expr, double conv, int maxiter, double *chi_2)
{
	int length, i, j, k, l, ndata, nparam, r1, r2, r3, c1, c2, c3, *lista, itst, itst1;
	symrec *tab1, *tab2, *csr, **parsym;
	AccRange *arx, *ary, *arz;
	double *x, *y, *z, currx, curry, currz, alamda, chisq, ochisq;
	double **covar, **alpha;
	char tmp_txt[500];

	if(d) curr_data = d;
	if(chi_2) *chi_2 = 0.0;
	txt_formula = expr;
	if(!curr_data || !par || !expr || !rx || !ry) return 0;
	//process ranges and create arrays
	arx = ary = arz = 0L;	x = y = z = 0L;	parval = 0L;	parsym = 0L;
	if(!(arx = new AccRange(rx)))return 0;
	i = arx->CountItems()+1;
	if(!(ary = new AccRange(ry))){
		delete arx;	return 0;
		}
	if(rz && !(arz = new AccRange(rz))){
		delete ary;	delete arx;	return 0;
		}
	if(!(x = (double*)malloc(i * sizeof(double)))){
		if(arz) delete arz;
		delete ary;	delete arx;	return 0;
		}
	if(!(y = (double*)malloc(i * sizeof(double)))){
		if(arz) delete arz;
		free(x);	delete arx;	delete ary;	return 0;
		}
	if(rz && !(z = (double*)malloc(i * sizeof(double)))){
		if(arz) delete arz;
		free(y);	free(x);	delete arx;	delete ary;	return 0;
		}
	arx->GetFirst(&c1, &r1);	ary->GetFirst(&c2, &r2);
	if(rz) arz->GetFirst(&c3, &r3);
	for(ndata = j = 0; j < i; j++) {
		if(rz) {
			if(arx->GetNext(&c1, &r1) && ary->GetNext(&c2, & r2) && arz->GetNext(&c3, &r3) &&
				curr_data->GetValue(r1, c1, &currx) && curr_data->GetValue(r2, c2, &curry) &&
				curr_data->GetValue(r3, c3, &currz)) {
				x[ndata] = currx;	y[ndata] = curry;	z[ndata] = currz;	ndata++;
				}
			}
		else {
			if(arx->GetNext(&c1, &r1) && ary->GetNext(&c2, & r2) &&
				curr_data->GetValue(r1, c1, &currx) && curr_data->GetValue(r2, c2, &curry)) {
				x[ndata] = currx;	y[ndata] = curry;	ndata++;
				}
			}
		}
	//common initialization for parser tasks
	push_parser();		//make code reentrant
	init_table();		length = strlen(*par);
	//process parameters
	if(!(buffer = (char*)malloc(length+2))){
		clear_table();	pop_parser();
		if(arz) delete arz;
		free(y);	free(x);	delete arx;	delete ary;
		return 0;
		}
	strcpy(buffer, *par);	buffer[length++] = ';';
	buffer[length] = 0;	buff_pos = 0;
	tab1 = sym_table;
	do {
		yyparse();
		}while(buff_pos < length);
	tab2 = sym_table;	free(buffer);	buffer =0L;
	for(nparam = 0, csr=tab2; csr != tab1; nparam++, csr = csr->next);
	parsym = (symrec**)malloc((nparam+1)*sizeof(symrec*));
	parval = (double**)malloc((nparam+1)*sizeof(double*));
	for(i = 0, csr=tab2; csr != tab1 && i < nparam; i++, csr = csr->next){
		parsym[i] = csr;	parval[i] = &csr->var;
		}
	//do iteratations to optimize fit
	lista = (int*)malloc(sizeof(int)*nparam);
	for(i = 0; i< nparam; i++) lista[i] = i;
	covar = dmatrix(1, nparam, 1, nparam);
	alpha = dmatrix(1, nparam, 1, nparam);
	alamda = -1.0;		itst = 0;
	mrqmin(x, y, z, ndata, parval, nparam, lista, nparam, covar, alpha, &chisq, fcurve, &alamda);
	if(!Check_MRQerror()) {
		for(itst = itst1 = 0, ochisq = chisq; itst < maxiter && chisq > conv && ochisq >= chisq && itst1 < 9; itst++) {
			ochisq = chisq;
			mrqmin(x, y, z, ndata, parval, nparam, lista, nparam, covar, alpha, &chisq, fcurve, &alamda);
			if(ochisq == chisq) itst1++;
			else itst1 = 0;
			}
		alamda = 0.0;
		mrqmin(x, y, z, ndata, parval, nparam, lista, nparam, covar, alpha, &chisq, fcurve, &alamda);
		Check_MRQerror();
		}
	for(i = nparam-1, j = k = l = 0; i >= 0; l = 0, i--) {
		if(k > 20) {
			if(tmp_txt[j-1] == ' ') j--;
			if(tmp_txt[j-1] == ';') j--;
			l = sprintf(tmp_txt+j, "\n");
			j += l;		k = 0;
			}
		l += sprintf(tmp_txt+j, "%s%s=%g;", j && k ? " " : "", parsym[i]->name, parsym[i]->GetValue());
		j += l;			k += l;
		}
	free(*par);	*par = strdup(tmp_txt);
	if(chi_2) *chi_2 = chisq;
	//write back spreadsheet data if necessary
	buffer = *par;	length = strlen(buffer);
	do {
		yyparse();
		}while(buff_pos < length);
	buffer = 0L;
	free_dmatrix(alpha, 1, nparam, 1, nparam);
	free_dmatrix(covar, 1, nparam, 1, nparam);
	if(arz) delete arz;		if(z) free(z);
	free(y);	free(x);	delete arx;	delete ary;
	if(parval) free(parval);	if(parsym) free(parsym);
	clear_table();
	pop_parser();
	if(curr_data){
		curr_data->Command(CMD_CLEAR_ERROR, 0L, 0L);
		curr_data->Command(CMD_REDRAW, 0L, 0L);
		}
	return itst < maxiter ? itst+1 : maxiter;
}









