// Copyright (c) 2000, 2001, 2002, 2003 by David Scherer and others.
// See the file license.txt for complete license terms.
// See the file authors.txt for a complete list of contributors.

#include "display.h"
#include "displaylist.h"
#include "mouseobject.h"
#include "kbobject.h"
#include "exceptions.h"
#include "gldevice.h"
#include <sstream>
#include <cmath>
#include <stdexcept>
#include <boost/python/class.hpp>
#include <boost/python/tuple.hpp>
#include <boost/python/return_internal_reference.hpp>
#include <boost/python/has_back_reference.hpp>

/************* Display implementation *************/

/*
 * Spew.  I hate global variables.  This bugger also make it harder to
 * initialize Gdk::Thread.
 * TODO: Get rid of them. 
 */
namespace {

visual::mutex allLock;

} // !namepspace anonymous

namespace visual {

// Static storage.	
std::vector<Display*> Display::all;
/* There is a requirement to initialize this variable with the first display that is created.
 * Traditionally, we have simply assigned 'this' to Display::selected in the constructor.
 * However, to use enable_shared_from_this<Display>::shared_from_this(), there must be at least
 * one owning shared_ptr to *this, which doesn't happen until *after* the constructor returns.
 * python class visual.ui.display provides a constructor that performs the required two-stage
 * construction.
 */
boost::shared_ptr<Display> Display::selected;

std::string 
Display::py_info( void) 
{
	return device->info();
}

void 
Display::py_select() 
{
	mutex::lock L(allLock);
	selected = shared_from_this();
	return;
}

void
Display::py_select( boost::python::object _self)
{
	mutex::lock L(allLock);
	selected = shared_from_this();
	self = _self;
	return;
}

vector 
Display::py_project( vector p) const
{
	tmatrix wct = device->get_wct();
	double w = wct.w(p);
	if (!w)
		return vector();

	p = (wct*p) / w;
	if (p.get_z() <= 0)
		return vector();

	return p;
}

Display::Display(Device& dev)
	: mouse(new mouseObject()),
	kb(new kbObject()),
	cursor(new cursorObject()),
	title("VPython"),
	title_changed(false),
	center(0,0,0),
	forward(0,0,-1),
	scale(0.1,0.1,0.1),
	up(0,1,0),
	foreground(1,1,1),
	background(0,0,0),
	fov(60*pi/180),
	uniform(true), 
	autoscale_enabled(true),
	autocenter_enabled(false),
	rotation_enabled(true),
	zoom_enabled(true),
	autoscale_max(1.0),
	autoscale_min(1.0),
	auto_show(true),
	region_valid(false),
	device(&dev)
{ 
	this->mouse->buttons = 0;

	device->display = this;
	device->onClose(true);

	mutex::lock A(allLock);
	displayID = all.size();
	all.push_back(this);
}

Display::Display()
	: mouse(new mouseObject()),
	kb(new kbObject()),
	cursor(new cursorObject()),
	title("VPython"),
	title_changed(false),
	center(0,0,0),
	forward(0,0,-1),
	scale(0.1,0.1,0.1),
	up(0,1,0),
	foreground(1,1,1),
	background(0,0,0),
	fov(60*pi/180),
	uniform(true), 
	autoscale_enabled(true),
	autocenter_enabled(false),
	rotation_enabled(true),
	zoom_enabled(true),
	autoscale_max(1.0),
	autoscale_min(1.0),
	auto_show(true),
	region_valid(false),
	device( new GLDevice())
{
	this->mouse->buttons = 0;

	device->display = this;
	device->onClose(true);

	mutex::lock A(allLock);
	displayID = all.size();
	all.push_back(this);
}

Display::~Display() 
{
	{
		mutex::lock A(allLock);
		all[displayID] = all.back();
		all[displayID]->displayID = displayID;
		all.pop_back();
	}
	if (device)
		delete(device);
}

void 
Display::shutdown() 
{
	std::vector<Display*> _all;
	{
		mutex::lock A(allLock);
		_all = all;
	}
	for( std::vector<Display*>::iterator i = _all.begin(); i != _all.end(); i++ ) {
		(*i)->device->hide();
		(*i)->device->join();
	}
}

void 
Display::waitclose() 
{
	std::vector<Display*> _all;
	{
		mutex::lock A(allLock);
		_all = all;
	}
	for( std::vector<Display*>::iterator i = _all.begin(); i != _all.end(); i++ ) {
		(*i)->device->join();
	}
} 

bool 
Display::allclosed() 
{
	std::vector<Display*> _all;
	{
		mutex::lock A(allLock);
		_all = all;
	}
	for( std::vector<Display*>::iterator i = _all.begin(); i != _all.end(); i++ ) {
		if (!(*i)->device->closed()) return false;
	}
	return true;
}

void
Display::internal_shutdown()
{
	// Order all windows to hide
	std::vector<Display*> _all;
	{
		mutex::lock A(allLock);
		_all = all;
	}
	for (std::vector<Display*>::iterator i = _all.begin(); i != _all.end(); ++i) {
		(*i)->device->get_context().cleanup();
		(*i)->device->hide();
	}
	threaded_exit(0);
} 

void 
Display::addObject() 
{
	if (auto_show)
		show();
}

void 
Display::show() 
{
	auto_show = false;
	device->show();
}

void 
Display::hide() 
{
	auto_show = false;
	device->hide();
	device->join();
}

double 
Display::get_tanfov() 
{
	return std::tan( fov*0.5 );
}

void 
Display::refreshCache() 
{
	if (region_valid) {
		vector extent = region_max - center;
		vector e2 = center - region_min;
		if (e2.x > extent.x) extent.x = e2.x;
		if (e2.y > extent.y) extent.y = e2.y;
		if (e2.z > extent.z) extent.z = e2.z;

		c_extent = extent;

		if (autocenter_enabled) {
			center = (region_min + region_max) * 0.5;
		}

		if (autoscale_enabled && !!extent) {
			if (uniform) {
				float mag = 1.0 / extent.mag();

				vector smax = vector(mag,mag,mag)*autoscale_max;
				vector smin = vector(mag,mag,mag)*autoscale_min;
	
				if (scale.mag() > smax.mag())
					scale = smax;
				if (scale.mag() < smin.mag())
					scale = smin;

			//scale = vector( mag, mag, mag );
			}
			else {
				vector s;
				s.x = extent.x ? 1.0 / extent.x : 0.0;
				s.y = extent.y ? 1.0 / extent.y : 0.0;
				s.z = extent.z ? 1.0 / extent.z : 0.0;

				if (scale.x > autoscale_max * s.x)
					scale.x = autoscale_max * s.x;
				if (scale.x < autoscale_min * s.x)
					scale.x = autoscale_min * s.x;
				if (scale.y > autoscale_max * s.y)
					scale.y = autoscale_max * s.y;
				if (scale.y < autoscale_min * s.y)
					scale.y = autoscale_min * s.y;
				if (scale.z > autoscale_max * s.z)
					scale.z = autoscale_max * s.z;
				if (scale.z < autoscale_min * s.z)
				scale.z = autoscale_min * s.z;
			}
		}
	}
	else
		c_extent = vector();

	c_uniform = uniform;
	c_rotation_enabled = rotation_enabled;
	c_zoom_enabled = zoom_enabled;
	c_title = title;
	c_title_changed = title_changed;
	title_changed = false;
	tanfov = get_tanfov();
	bgcolor = background;
	region_valid = false;

	c_forward = forward;
	c_up = up;
	c_center = center;

	vector Z = forward.norm();
	vector X = Z.cross( up ).norm();
	iview.x_column(X);
	iview.y_column(X.cross( Z ));
	iview.z_column(Z*-1.0);
	iview.w_column(Z*-1.0/tanfov);
	iview.w_row();
	view.invert_ortho(iview);

	model.x_column(scale.x,0,0);
	model.y_column(0,scale.y,0);
	model.z_column(0,0,scale.z);
	model.w_column(center.scale(scale)*-1);
	model.w_row();
  
	imodel.x_column(1.0 / model[0][0], 0, 0);
	imodel.y_column(0, 1.0 / model[1][1], 0);
	imodel.z_column(0, 0, 1.0 / model[2][2]);
	imodel.w_column(center);
	imodel.w_row();
}

    
vector
Display::get_extent()
{
	vector extent = region_max - center;
	vector e2 = center - region_min;
	if (e2.x > extent.x)
		extent.x = e2.x;
	if (e2.y > extent.y)
		extent.y = e2.y;
	if (e2.z > extent.z)
		extent.z = e2.z;
	return extent;
}

void
Display::exit_on_close( bool code)
{
	device->onClose( code);
}

void
Display::set_title( std::string new_title)
{
	write_lock L(mtx);
	title = new_title;
	title_changed = true;
}

std::string
Display::get_title() const
{
	return title;
}

vector
Display::get_range()
{
// 	read_lock L( mtx);
	return vector( scale.x ? 1.0/scale.x : 0.0,
			scale.y ? 1.0/scale.y : 0.0,
			scale.z ? 1.0/scale.z : 0.0 );
}


boost::python::list
Display::py_objects()
{
	boost::python::list ret;
	read_lock L( list_mutex);
	std::list< boost::shared_ptr<DisplayObject> >::iterator i;
	for ( i = objects.begin(); i != objects.end(); i++) {
		ret.append( (*i)->self);
	}
	return ret;
}

void
Display::set_forward( const vector& v)
{
	if (v.linear_multiple_of( up)) {
		throw std::invalid_argument( "forward may not be colinear with up.");
	}
	if (v == vector())
		throw std::invalid_argument( "forward may not be the zero vector.");
	write_lock L(mtx);
	forward = v;
}

void
Display::set_scale( const vector& v)
{
	write_lock L(mtx);
	autoscale_enabled = false;
	scale = v;
}

void
Display::set_up( const vector& v)
{
	if (v.linear_multiple_of( forward)) {
		throw std::invalid_argument( "up may not be colinear with forward.");
	}
	if (v == vector())
		throw std::invalid_argument( "up may not be the zero vector.");
	write_lock L(mtx);
	up = v;
}

void
Display::set_fov( const double& _fov)
{
	write_lock L(mtx);
	if (_fov == 0) {
		throw std::invalid_argument( "Orthogonal projection is not supported");
	} 
	if (_fov < 0 || _fov > M_PI) {
		throw std::invalid_argument( "fov must be between 0 and pi");
	}
	fov = _fov;
}

void
Display::set_uniform( bool _uniform)
{
	write_lock L(mtx);
	uniform = _uniform;
}

void
Display::set_autoscale( bool make_autoscale)
{
	if (make_autoscale) {
		write_lock L(mtx);
		autoscale_enabled = true;
	}
	else if (autoscale_enabled) {
		if (!device->closed()) { // Never *create* a display this way.
			device->frame();
			device->frame();
		}
		write_lock L(mtx);
		autoscale_enabled = false;
	}
}

void
Display::set_autocenter( bool make_autocenter)
{
	if (make_autocenter) {
		write_lock L(mtx);
		autocenter_enabled = true;
	}
	else if (autocenter_enabled) {
		if (!device->closed()) { // Never *create* a display this way.
			device->frame();
			device->frame();
		}
		write_lock L(mtx);
		autocenter_enabled = false;
	}			
}

void
Display::set_range( const vector& range)
{
	write_lock L(mtx);
	autoscale_enabled = false;
	
	scale.x = range.x ? 1.0/range.x : 0.0;
	scale.y = range.y ? 1.0/range.y : 0.0;
	scale.z = range.z ? 1.0/range.z : 0.0;
}

void
Display::set_visible( bool make_visible)
{
	if (make_visible) {
		this->show();
	}
	else {
		write_lock L(mtx);
		this->hide();
	}
}

void
Display::set_lighting( boost::python::list _lights)
{
	write_lock L(mtx);
	lights = lighting( lights, _lights);
}

void
Display::set_center( const vector& v)
{
	write_lock L(mtx);
	center = v;
}

void
Display::set_minscale( double scale)
{
	write_lock L(mtx);
	autoscale_min = scale;
}

void
Display::set_maxscale( double scale)
{
	write_lock L(mtx);
	autoscale_max = scale;
}

void
Display::set_userspin( bool spin)
{
	write_lock L(mtx);
	rotation_enabled = spin;
}

void
Display::set_userzoom( bool zoom)
{
	write_lock L(mtx);
	zoom_enabled = zoom;
}

void
Display::set_ambient( const float& l)
{
	write_lock L(mtx);
	lights.ambient = l;
}

void
Display::set_foreground( rgb fg)
{
	write_lock L(mtx);
	foreground = fg;
}

void
Display::set_background( rgb bg)
{
	write_lock L(mtx);
	background = bg;
}

void
Display::set_stereo( std::string mode)
{
	stereotype new_mode = NONE;
	if (mode == "nostereo")
		new_mode = NONE;
	else if (mode == "active")
		new_mode = ACTIVE;
	else if (mode == "passive")
		new_mode = PASSIVE;
	else if (mode == "redblue")
		new_mode = RED_BLUE;
	else if (mode == "redcyan")
		new_mode = RED_CYAN;
	else if (mode == "yellowblue")
		new_mode = YELLOW_BLUE;
	else if (mode == "greenmagenta")
		new_mode = GREEN_MAGENTA;
	else throw std::invalid_argument( "Unimplemented stereo mode");
	
	device->setStereo( new_mode);
}

std::string
Display::get_stereo() const
{
	stereotype mode = device->getStereo();
	switch (mode) {
		case NONE:
			return "nostereo";
		case ACTIVE:
			return "active";
		case PASSIVE:
			return "passive";
		case RED_BLUE:
			return "redblue";
		case RED_CYAN:
			return "redcyan";
		case GREEN_MAGENTA:
			return "greenmagenta";
		case YELLOW_BLUE:
			return "yellowblue";
		default:
			return "nostereo"; // We should never get here.
	}
}

bool
Display::get_fullscreen() const
{
	return device->getFullScreen();
}

void
Display::set_fullscreen( bool fs)
{
	device->setFullScreen( fs);
}

float
Display::get_stereodepth() const
{
	return device->getStereoDepth();
}

void
Display::set_stereodepth( float depth)
{
	device->setStereoDepth( depth);
}

void
Display::set_x( int x)
{
	write_lock L(mtx);
	device->setX( x);
}

void
Display::set_y( int y)
{
	write_lock L(mtx);
	device->setY( y);
}

void
Display::set_width( int width)
{
	write_lock L(mtx);
	device->setWidth( width);
}

void
Display::set_widthd( double width)
{
	this->set_width( static_cast<int>(width));
}

void
Display::set_height( int height)
{
	write_lock L(mtx);
	device->setHeight( height);
}

void
Display::set_heightd( double height)
{
	this->set_height( static_cast<int>( height));
}

int
Display::get_x()
{
	return device->getX();
}

int
Display::get_y()
{
	return device->getY();
}

int
Display::get_width()
{
	return device->getWidth();
}

int
Display::get_height()
{
	return device->getHeight();
}

boost::python::object
Display::get_selected(void)
{
	if (Display::selected)
		return Display::selected->getObject();
	else
		return boost::python::object();
}

boost::shared_ptr<kbObject>
Display::get_keyboard_object()
{
	if (auto_show)
		show();
	return kb;	
}

boost::shared_ptr<mouseObject>
Display::get_mouse()
{
	if (auto_show)
		show();
	return mouse;
}

void
display_init_type()
{
	using namespace boost::python;
	void (Display:: *select)(void) = &Display::py_select;
	void (Display:: *py_select)(object) = &Display::py_select;
	
	// x, y, width, and height may only be set at construction time.
	// All of the other constructor keyword arguments will be hybridized to python.
	class_<Display, boost::shared_ptr<Display>, boost::noncopyable> display_wrap( "display", init<>());
		// Properties of the display
		// The following properties are read/write
		display_wrap.add_property( "x", &Display::get_x, &Display::set_x)
		.add_property( "y", &Display::get_y, &Display::set_y)
		.add_property( "width", &Display::get_width, &Display::set_widthd)
		.add_property( "height", &Display::get_height, &Display::set_heightd)
		.add_property( "title", &Display::get_title, &Display::set_title)
		.def( "_get_forward", &Display::get_forward, return_internal_reference<>())
		.def( "_set_forward", &Display::set_forward)
		.def( "_get_scale", &Display::get_scale, return_internal_reference<>())
		.def( "_set_scale", &Display::set_scale)
		.def( "_get_up", &Display::get_up, return_internal_reference<>())
		.def( "_set_up", &Display::set_up)
		.add_property( "fov", &Display::get_fov, &Display::set_fov)
		.add_property( "uniform", &Display::is_uniform, &Display::set_uniform)
		.add_property( "foreground", &Display::get_foreground, &Display::set_foreground)
		.add_property( "background", &Display::get_background, &Display::set_background)
		.add_property( "autoscale", &Display::is_autoscale, &Display::set_autoscale)
		.add_property( "autocenter", &Display::is_autocenter, &Display::set_autocenter)
		.add_property( "userzoom", &Display::is_userzoom, &Display::set_userzoom)
		.add_property( "userspin", &Display::is_userspin, &Display::set_userspin)
		.def( "_get_range", &Display::get_range)
		.def( "_set_range", &Display::set_range_f) // Overload to set range from single float.
		.def( "_set_range", &Display::set_range)   // Standard interface to set range from vector.
		.add_property( "maxscale", &Display::get_maxscale, &Display::set_maxscale)
		.add_property( "minscale", &Display::get_minscale, &Display::set_minscale)
		.add_property( "visible", &Display::is_visible, &Display::set_visible)
		.add_property( "ambient", &Display::get_ambient, &Display::set_ambient)
		.add_property( "lights", &Display::get_lights, &Display::set_lighting)
		.def( "_get_center", &Display::get_center, return_internal_reference<>())
		.def( "_set_center", &Display::set_center)
		// The following properties are set-only
		// set-only data?  Are we sure?
		// .add_property( "exit")
		.def( "exit_on_close", &Display::exit_on_close)
		// .add_property( "newzoom")
		// The following properties are get-only
		.add_property( "mouse", &Display::get_mouse)
		.add_property( "kb", &Display::get_keyboard_object)
		.add_property( "cursor", &Display::get_cursor_object)
		.add_property( "objects", &Display::py_objects)
		.add_property( "extent", &Display::get_extent)
		.add_property( "stereo", &Display::get_stereo, &Display::set_stereo)
		.add_property( "stereodepth", &Display::get_stereodepth, &Display::set_stereodepth)
		.add_property( "fullscreen", &Display::get_fullscreen, &Display::set_fullscreen)
		.def( "show", &Display::show)
		.def( "hide", &Display::hide)
		.def( "info", &Display::py_info)
		.def( "select", select)
		.def( "select", py_select)
		.def( "project", &Display::py_project)
		.def( "get_selected", &Display::get_selected, 
			"Returns the active display.  May be None.")
		.staticmethod("get_selected")
		;
}

} // !namespace visual
