/* KInterbasDB Python Package - Header File for Core
**
** Version 3.1
**
** The following contributors hold Copyright (C) over their respective
** portions of code (see license.txt for details):
**
** [Original Author (maintained through version 2.0-0.3.1):]
**   1998-2001 [alex]  Alexander Kuznetsov   <alexan@users.sourceforge.net>
** [Maintainers (after version 2.0-0.3.1):]
**   2001-2002 [maz]   Marek Isalski         <kinterbasdb@maz.nu>
**   2002-2004 [dsr]   David Rushby          <woodsplitter@rocketmail.com>
** [Contributors:]
**   2001      [eac]   Evgeny A. Cherkashin  <eugeneai@icc.ru>
**   2001-2002 [janez] Janez Jere            <janez.jere@void.si>
*/

#ifndef _KINTERBASDB_H
#define _KINTERBASDB_H

#define KIDB_HOME_PAGE "http://sourceforge.net/projects/kinterbasdb"
#define KIDB_REPORT " -- please report this to the developers at "

/* The standard guide to Embedding and Extending says: "Since Python may
** define some pre-processor definitions which affect the standard headers
** on some systems, you must include Python.h before any standard headers
** are included." */
#include "Python.h"

/* 2002.12.31: Switched to memory allocator that redirects to standard in
** <= Py2.2, but is optimized for frequent small-size memory ops in Py2.3
** (this should be a good fit for kinterbasdb). */
#include "pymemcompat.h"

#include <time.h>
#include <math.h>

#include "ibase.h"

typedef unsigned char boolean;
  #define TRUE  1
  #define FALSE 0

#define COMPILER_IS_MSVC_WIN32 ( defined(_MSC_VER) && defined(WIN32) )
#define COMPILER_IS_BCPP_WIN32 ( defined(__BORLANDC__) && defined(__WIN32__) )

#define PYTHON_2_2_OR_LATER (PY_VERSION_HEX >= 0x020200F0)

/* 2003.03.31: */
#include "_kimem.h"


/* 2003.03.30: Python 2.1/2.2.0 compatibility: */
#ifndef PyBool_Check
  #define PyBool_FromLong(x)  PyInt_FromLong(x)
#endif

/* Inexplicably, PySequence_Fast_GET_SIZE was left out of Python 2.1. */
#ifndef PySequence_Fast_GET_SIZE
  #define PySequence_Fast_GET_SIZE(o) \
    (PyList_Check(o) ? PyList_GET_SIZE(o) : PyTuple_GET_SIZE(o))
#endif
/* PyObject_TypeCheck doesn't exist in Python 2.1, but a replacement is trivial
** because Python 2.1 does not allow type inheritance. */
#ifndef PyObject_TypeCheck
  #define PyObject_TypeCheck(o, tp) ((o)->ob_type == (tp))
#endif


/* Python 2.3b1 no longer defines LONG_LONG. */
#ifndef LONG_LONG
  #define LONG_LONG PY_LONG_LONG
#endif

#ifndef LONG_LONG_MIN
  #ifdef LLONG_MIN
    #define LONG_LONG_MIN LLONG_MIN
    #define LONG_LONG_MAX LLONG_MAX
  #else
    #ifdef _I64_MIN
      #define LONG_LONG_MIN _I64_MIN
      #define LONG_LONG_MAX _I64_MAX
    #endif
  #endif
#endif

/* 2004.04.21: */
/* FB 1.5 exposes ISC_SHORT in its header file, but FB 1.0 does not. */
#ifndef ISC_SHORT
  #define ISC_SHORT short
#endif


/* 2002.02.15: Detect Interbase 6 or later (including Firebird). */
#if (defined(SQL_DIALECT_CURRENT) && defined(SQL_DIALECT_V6))
  #define INTERBASE6_OR_LATER
  #define SQL_DIALECT_DEFAULT SQL_DIALECT_CURRENT
  #define SQL_DIALECT_OLDEST SQL_DIALECT_V5
#else
  #define SQL_DIALECT_DEFAULT 1
  #define SQL_DIALECT_OLDEST 1
  /* IB < 6.0 had less differentiated datetime types; SQL_TIMESTAMP is more
  ** representative of what columns called SQL_DATE actually contain. */
  #define SQL_TIMESTAMP SQL_DATE
  /* IB 6 API Guide p 301:  "The isc_decode_timestamp() is exactly the same as
  ** the isc_decode_date() function in versions of InterBase prior to 6.0." */
  #define isc_decode_timestamp  isc_decode_date
  /* IB 6 API Guide p 346:  "This call [isc_encode_timestamp()] is exactly the
  ** same as the older isc_encode_date(), which is still available for backward
  ** compatibility. */
  #define isc_encode_timestamp  isc_encode_date

  #define ISC_TIMESTAMP  ISC_QUAD
  #define blr_timestamp  blr_date

  /* 2004.07.18: IB < 6.0 doesn't define ISC_ULONG: */
  #define ISC_ULONG unsigned long
#endif

/* Beginning with Firebird 1.5, Firebird defines an integer constant FB_API_VER
** that we can use to determine the version of the client library we're
** compiling against.  Unfortunately, Firebird 1.0 defines no such thing, so
** we just test for an ad hoc definition that's defined in Firebird 1.0 but not
** Interbase 6.0. */
#ifdef dtype_null
  #define FIREBIRD_1_0_OR_LATER
#endif

#ifdef FB_API_VER
  #if FB_API_VER >= 15
    #define FIREBIRD_1_5_OR_LATER
  #endif
#endif /* def FB_API_VER? */


/* Field precision determination code uses DB fields not present before IB 6.0. */
#ifdef INTERBASE6_OR_LATER
  #define DETERMINE_FIELD_PRECISION
#endif


/******************** COMPILATION OPTIONS:BEGIN ********************/

/* Define KIDB_DEBUGGERING to get lots of debugging output to stderr.
** Note that this facility hasn't really been kept up to date; it's not very
** useful. */
/* #define KIDB_DEBUGGERING */

/* 2002.10.14: */
/* Change this definition to FALSE to turn off explicit GIL release and
** reacquisition.  If you plan on using events, turning this off is a very
** bad idea. */
#ifdef WITH_THREAD
  #define SHOULD_MANAGE_GIL    TRUE
#else
  #define SHOULD_MANAGE_GIL    FALSE
#endif /* WITH_THREAD */

/* 2003.01.20: */
/* For version 3.1, database event support is off by default; uncomment the
** line below to enable it (the prototype is only for Win32). */
#define ENABLE_DB_EVENT_SUPPORT


/* INITIAL_SQLVAR_CAPACITY is the number of XSQLVARs that a freshly allocated
** XSQLDA (such as the in_sqlda and out_sqlda of a new cursor) is configured
** to hold.
** If the initial capacity proves insufficient, more space is allocated
** automatically.  Therefore, INITIAL_SQLVAR_CAPACITY is an "optimization hint"
** rather than a "hard-coded limit". */
#define INITIAL_SQLVAR_CAPACITY        16


/******************** COMPILATION OPTIONS:END ********************/

/******************** HARD-CODED LIMITS:BEGIN ********************/

/* MAX_BLOB_SEGMENT_SIZE is an IB/Firebird engine constraint, not something
** we could overcome here in kinterbasdb.  For that matter, it's not a
** "constraint" in any meaningful sense, since a single blob can have many
** segments, and kinterbasdb manages the segmentation transparently. */
#define MAX_BLOB_SEGMENT_SIZE 65535

/* DSR:2002.02.15:
** I went through and tried to extract all hard-coded buffer size
** specifications to this HARD-CODED LIMITS section.  Generally, the respective
** sizes I chose are those used by the Interbase 6 API Guide, though it does
** not officially define the sizes, which leaves the looming specter of buffer
** overflows.
**
** As of 2003.02.13, I've made considerable effort to minimize the likelihood
** of buffer overflows; you can find notes about those measures in the source
** code where these constants are used. */

#define MAX_ISC_ERROR_MESSAGE_BUFFER_SIZE           1024

/* Firebird 1.5 and later publish the size of the status vector as
** ISC_STATUS_LENGTH. */
#ifdef ISC_STATUS_LENGTH
  #define STATUS_VECTOR_SIZE           ISC_STATUS_LENGTH
#else
  #define STATUS_VECTOR_SIZE                          20
#endif

#define DPB_BUFFER_SIZE                              256

#define ISC_INFO_BUFFER_SIZE                          20

#define ISC_INFO_RESULT_BUFFER_SIZE                 8192

/* 2003.02.19: */
/* These limits are based on the official API examples and observation that
** larger values crash the client library.  They are the size limits imposed
** on buffers incoming to pyob_attach_db. */
#define MAX_DPB_SIZE                                 255
#define MAX_DSN_SIZE                                 127


/* XSQLDA and XSQLVAR-related limits: */

/* As of 2003.02.13/FB1.0, attempting to prepare and execute a statement with
** an extremely large number of parameters causes the Firebird client library
** to to mishandle memory at risk of segfault.
** I have observed the problem to begin around 3625 parameters.
** kinterbasdb defines MAX_XSQLVARS_IN_SQLDA to prevent Python programmers
** from inadvertently running afoul of the FB client library's irresponsibility.
** If in the future the client library is fixed, there'll be no reason not to
** raise MAX_XSQLVARS_IN_SQLDA (though it goes without saying that anyone who
** actually uses SQL statements with > 2048 parameters routinely should be
** shot). */
/* YYY:2004.04.16: 2048 fields segfaulted on x86_64/1.5.1pre1, so I reduced
** the limit: */
#define MAX_XSQLVARS_IN_SQLDA     1024

/* The database engine does not allow more than 16 database handles to
** participate in a single distributed transaction. */
#define DIST_TRANS_MAX_DATABASES    16

/******************** HARD-CODED LIMITS:END ********************/

/******************** HANDY ABBREVIATIONS:BEGIN ********************/

/* 2002.12.24:  Returned to using multi-line macros instead of jamming
** everything onto a single line.  People using compilers incapable of dealing
** with multi-line macros will just have to deal with it. */

typedef enum {
  SIGN_NEGATIVE = -1,
  SIGN_POSITIVE =  1
} NumberSign;

typedef enum {
  SUBTYPE_NONE    = 0,
  SUBTYPE_NUMERIC = 1,
  SUBTYPE_DECIMAL = 2
} SQLSubtype;

typedef enum {
  SQLIND_NULL     = -1,
  SQLIND_NOT_NULL =  0
} SQLIndicator_State;


/* See IB6 API Guide page 92. */
#define XSQLVAR_IS_ALLOWED_TO_BE_NULL(sqlvar) \
  ((sqlvar->sqltype & 1) != 0)

#define XSQLVAR_IS_NULL(sqlvar) \
  (*(sqlvar->sqlind) == SQLIND_NULL)

#define XSQLVAR_SQLTYPE_IGNORING_NULL_FLAG(sqlvar) \
  (sqlvar->sqltype & ~1)

#define XSQLVAR_SQLTYPE_READ_NULL_FLAG(sqlvar) \
  (sqlvar->sqltype & 1)

#define XSQLVAR_SET_NULL(sqlvar) \
  (*(sqlvar->sqlind) = SQLIND_NULL)

#define XSQLVAR_SET_NOT_NULL(sqlvar) \
  (*(sqlvar->sqlind) = SQLIND_NOT_NULL)

/* Macros for printing debugging information to stderr. */
#ifdef KIDB_DEBUGGERING
  #define debug_printf( x ) fprintf( stderr, ( x ) )
  #define debug_printf1( x, y ) fprintf( stderr, ( x ), ( y ) )
  #define debug_printf2( x, y, z ) fprintf( stderr, ( x ), ( y ), ( z ) )

  #define debug_dump_status_vector(x) dumpStatusVector(x)
#else
  #define debug_printf( x )
  #define debug_printf1( x, y )
  #define debug_printf2( x, y, z )

  #define debug_dump_status_vector(x)
#endif /* KIDB_DEBUGGERING */

/* DB_API_ERROR acts like a boolean function, returning TRUE if
** status_vector indicates an error. */
#define DB_API_ERROR(status_vector) \
  ( ((status_vector[0] == 1) && status_vector[1] > 0) ? TRUE : FALSE )


#define MIN( a, b ) ( ( a < b ) ? a : b )
#define MAX( a, b ) ( ( a > b ) ? a : b )

#define RETURN_PY_NONE \
  Py_INCREF(Py_None); \
  return Py_None;


/******************** HANDY ABBREVIATIONS:END ********************/

/******************** MODULE TYPE DEFINITIONS:BEGIN ********************/
/* See Appendix B of Interbase 6 API Guide for explanation of Interbase API
** structures. */
/* 2002.12.15: declared as extern, not staticforward, so that they're visible
** to _kievents. */
extern PyTypeObject ConnectionType;
extern PyTypeObject CursorType;
/* 2003.04.27: */
extern PyTypeObject TransactionHandleType;

/* 2002.06.09: */
/* This structure supports a moderately ugly approach to determining the
** precision of a given field (querying the system tables). */
typedef struct {
  isc_stmt_handle stmt_handle_table;
  isc_stmt_handle stmt_handle_stored_procedure;

  XSQLDA *in_da;
  XSQLDA *out_da;

  XSQLVAR *out_var;

  PyObject *result_cache;
} CursorDescriptionCache;


typedef enum {
  CONNECTION_STATE_CLOSED = 0,
  CONNECTION_STATE_OPEN = 1
} ConnectionState;


typedef struct { /* definition of type ConnectionObject */
  PyObject_HEAD /* Python API - infrastructural macro. */

  unsigned short   dialect;
  isc_db_handle    db_handle;

  isc_tr_handle    trans_handle;
  /* 2003.04.27: */
  PyObject *group;

  /* Buffer used by Interbase API to store error status of calls. */
  ISC_STATUS       status_vector[ STATUS_VECTOR_SIZE ];

  ConnectionState _state; /* Connection state flag. */

  #ifdef DETERMINE_FIELD_PRECISION
  CursorDescriptionCache *desc_cache;
  #endif

  /* 2003.03.30: */
  PyObject *type_trans_in;
  PyObject *type_trans_out;

  /* 2003.10.16: */
  PyObject *output_type_trans_return_type_dict;
} ConnectionObject;


#define CONN_REQUIRE_OPEN(connection) \
  if ( _conn_require_open(connection, NULL) != 0 ) \
    return NULL;

#define CONN_REQUIRE_OPEN2(connection, failure_message) \
  if ( _conn_require_open(connection, failure_message) != 0 ) \
    return NULL;


typedef struct {
  short sqltype;
  short sqllen;
} OriginalXSQLVARSpecificationCache;

typedef enum {
  CURSOR_STATE_CLOSED = 0,
  CURSOR_STATE_OPEN = 1
} CursorState;

typedef struct { /* definition of type CursorObject */
  PyObject_HEAD /* Python API - infrastructural macro. */

  /* Connection associated with this cursor. */
  ConnectionObject           *connection;

  /* Interbase API - statement handle. */
  isc_stmt_handle             stmt_handle;
  /* Interbase API - containers for input and result parameter structures. */
  XSQLDA                     *in_sqlda;
  XSQLDA                     *out_sqlda;

  /* 2002.10.07: */
  /* Added to support the use of named cursors with "SELECT ... FOR UPDATE"
  ** syntax. */
  PyObject                   *name;

  /* 2002.08.22: */
  /* The in_var_orig_spec cache variable allows us to record the original
  ** values supplied by the database engine for each XSQLVAR's sqltype and
  ** sqllen fields.  These values are reset in cases of implicit parameter
  ** conversion (or in *any* case with CHAR/VARCHAR fields), but the original
  ** settings must be remembered (see _prepare_statement_if_necessary in
  ** _kinterbasdb.c)
  ** !AND! restored (see PyObject2XSQLDA in _kiconversion.c) before attempting
  ** to convert each incoming parameter list.
  **
  **   For example, suppose a VARCHAR field is defined to be 20 characters at
  ** most.  Suppose the first row of input parameters to an insert statement
  ** supplies a string value that is only 15 characters long.  In this case,
  ** the incoming parameter conversion function (PyObject2XSQLVAR) does the
  ** following:
  ** 1. sets the XSQLVAR's sqltype flag to SQL_TEXT (was SQL_VARYING)
  ** 2. sets the XSQLVAR's sqldata pointer to point to the internal buffer
  **   of the Python string (was NULL)
  ** 3. sets the XSQLVAR's sqllen field to indicate the length of that
  **   specific value, which is 15 (was the maximum length of the field, 20).
  **
  **   Suppose that the next row of input parameters includes a 17-character
  ** string for the same field.  Unless the original sqllen flag (20) has been
  ** restored, the range checking code will conclude that the field has a
  ** maximum length of 15 characters (which just happened to be the lentgh of
  ** the last value, now stored in the XSQLVAR's sqllen flag).
  **   With the XSQLVAR's original sqltype and sqllen flags restored to their
  ** pre-implicit-conversion values (sqltype SQL_VARYING and sqllen 20, in the
  ** running example), everything proceeds fine.
  **   I smell a really ugly hack, don't you?  Perhaps I should've rejected
  ** the implicit parameter conversion feature request, if my implementation
  ** of can be no cleaner than this.
  */
  OriginalXSQLVARSpecificationCache *in_var_orig_spec;

  /* Utility buffer where query result values reside before they're converted
  ** to Python values. */
  char                       *out_buffer;

  /* Variables to support caching the last SQL string executed (in order to
  ** detect whether a new statement is identical and thereby avoid having to
  ** reprepare, etc.). */
  PyObject                   *previous_sql;

  /* 2003.01.26: */
  int                        statement_type;
  #define                    NULL_STATEMENT_TYPE -1

  /* 2003.09.06b: */
  PyObject                   *objects_to_release_after_execute;

  /* 2002.02.21:
  ** Variable to support caching a single row of results from a stored
  ** procedure invoked using EXECUTE PROCEDURE syntax.  This approach is
  ** exceptionaly clumsy, but necessary in order to deal with of Interbase
  ** API quirks and still comply with the Python DB API 2.0.
  ** (See bug #520793.)
  ** (See special cases involving isc_info_sql_stmt_exec_procedure in
  ** functions pyob_execute and pyob_fetch.) */
  PyObject                   *exec_proc_results;

  /* 2002.04.27:
  ** last_fetch_status caches the return code of the last isc_dsql_fetch call
  ** made with this cursor, in order to assist in bridging a gap between the
  ** IB API and the Python DB API.
  **   The Python DB API requires that extraneous fetches after a result set
  ** has been exhausted be tolerated, whereas the IB API raises an error in
  ** such cases.  This flag allows kinterbasdb to detect this situation and
  ** handle it in the Pythonic way, instead of calling isc_dsql_fetch and
  ** encountering an error. */
  int last_fetch_status;

  /* Buffer used by Interbase API to store error status of calls. */
  ISC_STATUS                  status_vector[ STATUS_VECTOR_SIZE ];

  CursorState _state; /* Cursor state flag. */

  /* 2003.03.30: */
  PyObject *type_trans_in;
  PyObject *type_trans_out;

  /* 2003.10.16: */
  PyObject *output_type_trans_return_type_dict;
} CursorObject;

#define CUR_REQUIRE_OPEN(cursor) \
  if ( _cur_require_open(cursor, NULL) != 0 ) \
    return NULL;

#define CUR_REQUIRE_OPEN2(cursor, failure_message) \
  if ( _cur_require_open(cursor, failure_message) != 0 ) \
    return NULL;


/* 2003.04.27: support for distributed transactions */

/* ISC_TEB:  See IB6 API Guide page 71. */
typedef struct teb {
  long *db_ptr;
  long tpb_len;
  char *tpb_ptr;
} ISC_TEB;

typedef struct { /* definition of type TransactionHandleObject */
  PyObject_HEAD /* Python API - infrastructural macro. */

  isc_tr_handle native_handle;
} TransactionHandleObject;


/******************** MODULE TYPE DEFINITIONS:END ********************/

/******************** FUNCTION PROTOTYPES:BEGIN ********************/

void close_cursor(CursorObject *cursor);

#ifndef _MSC_VER
void init_kinterbasdb( void );
#endif /* _MSC_VER */

#ifdef KIDB_DEBUGGERING
void dumpXSQLDA(XSQLDA *sqlda);
void dumpXSQLVAR(XSQLVAR *sqlvar);
void dumpStatusVector(ISC_STATUS *sv);
#endif /* KIDB_DEBUGGERING */


/* Exception functions moved the _kinterbasdb.c on 2003.01.01. */
extern void raise_sql_exception( PyObject *exc, const char *where,
    ISC_STATUS pvector[]
  );

extern void raise_exception( PyObject *exc, const char *description );

/******************** FUNCTION PROTOTYPES:END ********************/


/* Allow other source files that are compiled in the kinterbasdb core unit
** to access these MIN/MAX global variables: */
extern PyObject *SHRT_MIN_As_PyObject;
extern PyObject *SHRT_MAX_As_PyObject;

extern PyObject *INT_MIN_As_PyObject;
extern PyObject *INT_MAX_As_PyObject;

extern PyObject *LONG_MIN_As_PyObject;
extern PyObject *LONG_MAX_As_PyObject;

extern PyObject *LONG_LONG_MIN_As_PyObject;
extern PyObject *LONG_LONG_MAX_As_PyObject;

#endif /* not def _KINTERBASDB_H */
