/*
 * libsysactivity
 * http://sourceforge.net/projects/libsysactivity/
 * Copyright (c) 2009, 2010 Carlos Olmedo Escobar <carlos.olmedo.e@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef SA_MEMORY
#warning You should not include this file directly from your program.
#endif

/**
 * \defgroup memory Memory interface
 * @{
 */
#ifndef SA_COMMON_MEMORY_H_
#define SA_COMMON_MEMORY_H_

/** \struct sa_memory memory.h
 * This structure gathers the details about the memory usage. Each value is measured in bytes.
 */
SA_EXPORT struct sa_memory {
#ifdef SA_MEMORY_TOTAL
	uint64_t total; //!< Total amount of RAM available in the system.
#endif
#ifdef SA_MEMORY_FREE
	uint64_t free; //!< Unused amount of memory.
#endif
#ifdef SA_MEMORY_ACTIVE
	uint64_t active; //!< Amount of memory in use or recently used.
#endif
#ifdef SA_MEMORY_INACTIVE
	uint64_t inactive; //!< Amount of memory that has not been recently used.
#endif
#ifdef SA_MEMORY_BUFFERS
	uint64_t buffers; //!< Memory used for cached files. Useless for metrics nowadays.
#endif
#ifdef SA_MEMORY_SWAP_TOTAL
	uint64_t swap_total; //!< Total amount of swap present in the system.
#endif
#ifdef SA_MEMORY_SWAP_FREE
	uint64_t swap_free; //!< Free amount of swap available.
#endif
#ifdef SA_MEMORY_SWAP_CACHED
	uint64_t swap_cached; //!< Amount of swap that is cached.
#endif
#ifdef SA_MEMORY_WIRED
	uint64_t wired; //!< Memory not placed on any queue.
#endif
#ifdef SA_MEMORY_CACHED
	uint64_t cached; //!< Amount of cached memory.
#endif
#ifdef SA_MEMORY_DIRTY
	uint64_t dirty; //!< Amount of memory waiting for been written to the disk.
#endif
#ifdef SA_MEMORY_EXECUTABLE
	uint64_t executable; //!< Memory used to hold executable data
#endif
#ifdef SA_MEMORY_FILES
	uint64_t files; //!< Memory used by cached file data
#endif
#ifdef SA_MEMORY_LOCKED
	uint64_t locked; //!< Memory that can not be moved to swap
#endif
};

/** \struct sa_swap memory.h
 * This structure gathers the details about a swap device. Each value is measured in bytes.
 */
SA_EXPORT struct sa_swap {
#ifdef SA_SWAP_NAME
	char name[SA_SWAP_NAME]; //!< The name of the swap device.
#endif
#ifdef SA_SWAP_TOTAL
	uint64_t total; //!< Total amount of swap in this device.
#endif
#ifdef SA_SWAP_FREE
	uint64_t free; //!< Available amount of swap in this device.
#endif
#ifdef SA_SWAP_TYPE
	int8_t type; //!< The type of swap media. If it's in a device the value is 1. If it's based in a file the value is 2.
#endif
};

#ifdef SA_OPEN_MEMORY
/**
 * Prepares the resources needed for retrieving memory statistics. This function exists (and is needed) only when SA_OPEN_MEMORY is defined.
 * @return If successful 0 is returned, otherwise an error code is returned.  If the operative system is not supported the return value will be ENOTSUP.
 * @see sa_close_memory()
 */
SA_EXPORT int sa_open_memory(void);
#endif

#ifdef SA_CLOSE_MEMORY
/**
 * This function closes the resources used for retrieving memory statistics. You should call it even when there was a previous error in another function of this API. This function exists (and is needed) only when SA_CLOSE_MEMORY is defined.
 * @return If successful 0 is returned, otherwise an error code is returned.
 * @see sa_open_memory()
 */
SA_EXPORT int sa_close_memory(void);
#endif

/**
 * Retrieves statistics about the usage of the memory.
 * @param dst Where the statistics will be stored.
 * @return If successful 0 is returned, otherwise an error code is returned.
 */
SA_EXPORT int sa_get_memory(struct sa_memory* dst);

/**
 * Gives the total number of swap file systems.
 * @param number The number will be stored here
 * @return If successful 0 is returned, otherwise an error code is returned.
 */
SA_EXPORT int sa_count_swaps(uint16_t* number);

/**
 * Retrieves statistics from a given swap index.
 * @param index The swap index. It starts from 0.
 * @param dst Where the statistics will be stored.
 * @return If successful 0 is returned, otherwise an error code is returned. ENODEV is returned when the requested swap index was out of range.
 */
SA_EXPORT int sa_get_swap(uint16_t index, struct sa_swap* dst);

/**
 * Retrieves statistics from as many swap fs as possible.
 * @param dst A buffer where the statistics will be stored.
 * @param dst_size The number of swap fs that fits in the dst buffer. If it's not big enough dst will be filled but ENOMEM will be returned.
 * @param written The amount of swap fs statistics written.
 * @return If successful 0 is returned, otherwise an error code is returned.
 */
SA_EXPORT int sa_get_swaps(struct sa_swap* dst, uint16_t dst_size, uint16_t* written);

/*@}*/
#endif /* SA_COMMON_MEMORY_H_ */
