/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2005  Armin Bauer <armin.bauer@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; version 
 * 2.1 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */

#include <libsyncml/syncml.h>
#include <libsyncml/syncml_internals.h>

#ifdef ENABLE_HTTP
#include <libsyncml/sml_error_internals.h>
#include <libsyncml/sml_transport_internals.h>

#include "http_server.h"
#include "http_server_internals.h"

/**
 * @defgroup GroupIDPrivate Group Description Internals
 * @ingroup ParentGroupID
 * @brief The private part
 * 
 */
/*@{*/

static void _server_callback(SoupServerContext *context, SoupMessage *msg, gpointer data)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, context, msg, data);
	SmlMimeType mimetype = SML_MIMETYPE_UNKNOWN;
	smlAssert(data);
	SmlTransportHttpServerEnv *env = data;
	SmlError *error = NULL;
	
	char *path = soup_uri_to_string (soup_message_get_uri(msg), TRUE);
	smlTrace(TRACE_INTERNAL, "%s %s HTTP/1.%d", msg->method, path, soup_message_get_http_version(msg));
	
	if (soup_message_get_http_version(msg) != 1) {
		smlErrorSet(&error, SML_ERROR_NOT_SUPPORTED, "Wrong http version");
		soup_message_set_status(msg, SOUP_STATUS_NOT_IMPLEMENTED);
		goto error;
	}

	if (soup_method_get_id(msg->method) != SOUP_METHOD_ID_POST) {
		smlErrorSet(&error, SML_ERROR_NOT_SUPPORTED, "Wrong method");
		soup_message_set_status(msg, SOUP_STATUS_NOT_IMPLEMENTED);
		goto error;
	}
	
	if (env->url && g_strcasecmp(path, env->url)) {
		smlErrorSet(&error, SML_ERROR_FILE_NOT_FOUND, "Not Found");
		soup_message_set_status(msg, SOUP_STATUS_NOT_FOUND);
		goto error;
	}
	g_free(path);
	
	const char *header = soup_message_get_header(msg->request_headers, "Content-Type");
	if (header && !g_strncasecmp(header, SML_ELEMENT_XML, strlen(SML_ELEMENT_XML)))
		mimetype = SML_MIMETYPE_XML;
	else if(header && !g_strncasecmp(header, SML_ELEMENT_WBXML, strlen(SML_ELEMENT_WBXML)))
		mimetype = SML_MIMETYPE_WBXML;
	else if(header && !g_strncasecmp(header, SML_ELEMENT_SAN, strlen(SML_ELEMENT_SAN)))
		mimetype = SML_MIMETYPE_SAN;
	else if (header) {
		smlErrorSet(&error, SML_ERROR_GENERIC, "Unknown mimetype");
		soup_message_set_status (msg, SOUP_STATUS_BAD_REQUEST);
		goto error;
	} else {
		smlErrorSet(&error, SML_ERROR_GENERIC, "Faulty mimetype");
		soup_message_set_status (msg, SOUP_STATUS_BAD_REQUEST);
		goto error;
	}
	
	if (!msg->request.length) {
		/* Tolerate empty message for now. Http client transport could send empty messages
		   for initialization. Throwing an error here breaks testcase: manager_start_session 
		   Empty Message for initialization got introduced with r251.
		   It's not fixed yet if r251 will be the final implementation for http client initalization. */
		soup_message_io_pause(msg);
		smlTrace(TRACE_EXIT, "%s (empty message! tolerated for now ...)", __func__);
		return;

		/* FIXME: Throw an error again if changes in r251 got clarified/fixed. */
		/*
		smlErrorSet(&error, SML_ERROR_GENERIC, "No data sent");
		soup_message_set_status (msg, SOUP_STATUS_BAD_REQUEST);
		goto error;
		*/
	}
	
	SmlLink *link = smlLinkNew(env->tsp, msg, &error);
	if (!link)
		goto error;

	// FIXME: this is unnecessary because soup_message_io_pause ensure this alreay
	// g_object_ref(msg);
	
	SmlTransportData *tspdata = smlTransportDataNew(msg->request.body, msg->request.length, mimetype, FALSE, &error);
	if (!tspdata)
		goto error_unref_msg;
	
	smlTransportReceiveEvent(env->tsp, link, SML_TRANSPORT_EVENT_DATA, tspdata, NULL);
	
	smlLinkDeref(link);
	smlTransportDataDeref(tspdata);
	
	soup_message_io_pause(msg);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return;

error_unref_msg:
	// FIXME: this is unnecessary because libsoup's io_unpause makes the cleanup
	// g_object_unref(msg);
	smlLinkDeref(link);
error:
	soup_server_message_set_encoding (SOUP_SERVER_MESSAGE (msg), SOUP_TRANSFER_CONTENT_LENGTH);
	soup_message_io_unpause(msg);
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(&error));
	smlErrorDeref(&error);
}

/*@}*/

/**
 * @defgroup GroupID Group Description
 * @ingroup ParentGroupID
 * @brief What does this group do?
 * 
 */
/*@{*/

static void *smlTransportHttpServerInit(SmlTransport *tsp, const void *data, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, tsp, data, error);
	smlAssert(tsp);
	smlAssert(data);
	
	g_type_init();
	
	SmlTransportHttpServerEnv *env = smlTryMalloc0(sizeof(SmlTransportHttpServerEnv), error);
	if (!env)
		goto error;
	
	const SmlTransportHttpServerConfig *config = data;
	
	if (!(config->port > 0 && config->port < 65535)) {
		smlErrorSet(error, SML_ERROR_MISCONFIGURATION, "specified port was wrong");
		goto error_free_env;
	}
	env->port = config->port;
	
	// FIXME: do url checking. use whitelist
	env->url = g_strdup(config->url);
	env->interface = g_strdup(config->interface);
	
	smlTrace(TRACE_INTERNAL, "config: port %i, url %s, interface %s", config->port, config->url, config->interface);
	
	env->tsp = tsp;

	smlTrace(TRACE_INTERNAL, "http server uses context %p.", tsp->context);
	if(!config->ssl_key || !config->ssl_crt) {
		env->server = soup_server_new(SOUP_SERVER_ASYNC_CONTEXT, tsp->context, SOUP_SERVER_PORT, env->port, NULL);
	} else {
		env->server = soup_server_new(SOUP_SERVER_ASYNC_CONTEXT, tsp->context, SOUP_SERVER_PORT, env->port, SOUP_SERVER_SSL_CERT_FILE, config->ssl_crt, SOUP_SERVER_SSL_KEY_FILE, config->ssl_key, NULL);
	}
	if (!env->server) {
		smlErrorSet(error, SML_ERROR_MISCONFIGURATION, "Unable to spawn server");
		goto error_free_env;
	}
	
	soup_server_add_handler(env->server, NULL, NULL, _server_callback, NULL, env);
	
	soup_server_run_async(env->server);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return env;

error_free_env:
	g_free(env->url);
	g_free(env->interface);
	g_free(env);
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return NULL;
}

static SmlBool smlTransportHttpServerFinalize(void *data, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p)", __func__, data, error);
	smlAssert(data);
	SmlTransportHttpServerEnv *env = data;
	
	smlAssert(env->tsp);
	
	soup_server_quit(env->server);
	// g_object_unref(env->server);
	
	g_free(env->url);
	g_free(env->interface);
	g_free(env);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;
}

static void smlTransportHttpServerSend(void *userdata, void *linkdata, SmlTransportData *data, SmlError *error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p, %p)", __func__, userdata, linkdata, data, error);
	smlAssert(userdata);
	smlAssert(linkdata);
	smlAssert(data);
	SmlTransportHttpServerEnv *env = userdata;
	smlAssert(env);
	
	SoupMessage *msg = linkdata;
	
	if (error)
		goto error_free_message;

	// duplicate the data because sometimes data is freed to early
	char *soupcopy = (char *) smlTryMalloc0(data->size, &error);
	if (error)
		goto error_free_message;
	memcpy(soupcopy, data->data, data->size);

	soup_message_set_status (msg, SOUP_STATUS_OK);
	soup_server_message_set_encoding (SOUP_SERVER_MESSAGE(msg), SOUP_TRANSFER_CONTENT_LENGTH);

	switch (data->type) {
		case SML_MIMETYPE_XML:
			soup_message_add_header(msg->response_headers, "Accept", "application/vnd.syncml+xml");
			soup_message_set_response (msg, "application/vnd.syncml+xml", SOUP_BUFFER_SYSTEM_OWNED,
						soupcopy, data->size);
			break;
		case SML_MIMETYPE_WBXML:
			soup_message_add_header(msg->response_headers, "Accept", "application/vnd.syncml+wbxml");
			soup_message_set_response (msg, "application/vnd.syncml+wbxml", SOUP_BUFFER_SYSTEM_OWNED,
						soupcopy, data->size);
			break;
		default:
			smlErrorSet(&error, SML_ERROR_GENERIC, "Unknown Mimetype");
			goto error_free_message;
	}
	
	soup_message_io_unpause(msg);

	// FIXME: the documentation of libsoup never show an example with such a call
	// FIXME: msg is perhaps in use by libsoup because this is asynchronous I/O
	// g_object_unref(msg);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return;
	
error_free_message:
	if (smlErrorGetClass(&error) <= SML_ERRORCLASS_RETRY)
		soup_message_set_status_full(msg, SOUP_STATUS_BAD_REQUEST, smlErrorPrint(&error));
	else
		soup_message_set_status_full(msg, SOUP_STATUS_INTERNAL_SERVER_ERROR, smlErrorPrint(&error));

	soup_server_message_set_encoding (SOUP_SERVER_MESSAGE (msg), SOUP_TRANSFER_CONTENT_LENGTH);
	soup_message_io_unpause(msg);

	// g_object_unref(msg);
	smlErrorDeref(&error);
	smlTrace(TRACE_EXIT, "%s: Sent Error", __func__);
	return;
}

/** @brief Function description
 * 
 * @param parameter This parameter does great things
 * @returns What you always wanted to know
 * 
 */
SmlBool smlTransportHttpServerNew(SmlTransport *tsp, SmlError **error)
{
	tsp->functions.initialize = smlTransportHttpServerInit;
	tsp->functions.finalize = smlTransportHttpServerFinalize;
	tsp->functions.send = smlTransportHttpServerSend;
	
	return TRUE;
}

#endif //ENABLE_HTTP

/*@}*/
