/*
 * Copyright (C) The MX4J Contributors.
 * All rights reserved.
 *
 * This software is distributed under the terms of the MX4J License version 1.0.
 * See the terms of the MX4J License in the documentation provided with this software.
 */

package mx4j.tools.adaptor.rmi;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.rmi.registry.Registry;
import java.rmi.server.RemoteObject;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import javax.management.MBeanRegistration;
import javax.management.MBeanServer;
import javax.management.ObjectName;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import mx4j.log.Log;
import mx4j.log.Logger;
import mx4j.tools.adaptor.interceptor.AdaptorInterceptor;
import mx4j.tools.adaptor.interceptor.ContextClassLoaderAdaptorInterceptor;
import mx4j.tools.adaptor.interceptor.Interceptor;
import mx4j.tools.adaptor.interceptor.Invocation;
import mx4j.tools.adaptor.interceptor.InvocationResult;
import mx4j.tools.adaptor.interceptor.InvokerAdaptorInterceptor;
import mx4j.tools.adaptor.interceptor.LoggerAdaptorInterceptor;
import mx4j.tools.connector.JMXAddress;
import mx4j.tools.connector.MalformedJMXAddressException;

/**
 * The RMI adaptor MBean, base for the JRMP and IIOP protocol
 *
 * @version $Revision: 1.3 $
 */
public abstract class RMIAdaptor extends RemoteObject implements RMIAdaptorMBean, MBeanRegistration, RemoteAdaptor
{
   private MBeanServer m_server;
   private ObjectName m_objectName;
   private ArrayList m_interceptors = new ArrayList();
   private ArrayList m_interceptorNames = new ArrayList();
   private JMXAddress m_address;
   private Interceptor m_headInterceptor;

   //
   // Implementation of MBeanRegistration
   //

   public ObjectName preRegister(MBeanServer server, ObjectName name) throws Exception
   {
      m_server = server;
      m_objectName = name;
      return name;
   }

   public void postRegister(Boolean registrationDone)
   {
   }

   public void preDeregister() throws Exception
   {
      stop();
   }

   public void postDeregister()
   {
   }

   //
   // Implementation of the management interface
   //

   /**
    * Starts this adaptor, so that it can accept incoming calls
    *
    * @see #stop
    * @see #isRunning
    */
   public abstract void start() throws Exception;

   /**
    * Stops this adaptor, so that it does not accept incoming calls anymore
    *
    * @see #start
    */
   public abstract void stop() throws Exception;

   /**
    * Returns whether this adaptor has been started and not been stopped.
    *
    * @see #stop
    */
   public abstract boolean isRunning();

   /**
    * Returns the protocol of this adaptor
    */
   public abstract String getProtocol();

   /**
    * Returns the JNDI name under which this RMI Adaptor is registered
    */
   public String getJNDIName()
   {
      return m_address == null ? null : m_address.getPath();
   }

   /**
    * Sets the JNDI name under which the RMI adaptor should be registered. <br>
    * This method can be called only if this adaptor is not running.
    */
   public void setJNDIName(String name)
   {
      if (name == null) throw new IllegalArgumentException("JNDI name cannnot be null");
      if (name.trim().length() == 0) throw new IllegalArgumentException("JNDI name cannnot be empty");

      if (!isRunning())
      {
         try
         {
            if (m_address == null)
            {
               m_address = new JMXAddress("rmi", "localhost", Registry.REGISTRY_PORT, name);
            }
            else
            {
               JMXAddress address = new JMXAddress(m_address.getProtocol(), m_address.getHost(), m_address.getPort(), name);
               Map properties = m_address.getProperties();
               for (Iterator i = properties.entrySet().iterator(); i.hasNext();)
               {
                  Map.Entry entry = (Map.Entry)i.next();
                  address.putProperty((String)entry.getKey(), entry.getValue());
               }
               // Replace the address
               m_address = address;
            }
         }
         catch (MalformedJMXAddressException ignored)
         {
         }
      }
      else
      {
         throw new IllegalStateException("Cannot perform this operation while running");
      }
   }

   /**
    * Puts a JNDI property in the environment for the JNDI Initial Context used by this adaptor. <br>
    * This method can be called only if this adaptor is not running.
    *
    * @see #clearJNDIProperties
    */
   public void putJNDIProperty(Object property, Object value)
   {
      putNamingProperty(property, value);
   }

   /**
    * @deprecated Replaced by {link #putJNDIProperty}
    */
   public void putNamingProperty(Object property, Object value)
   {
      if (!isRunning())
      {
         try
         {
            if (Context.PROVIDER_URL.equals(property))
            {
               if (m_address == null)
               {
                  m_address = new JMXAddress("jmx:" + value);
               }
               else
               {
                  Map properties = m_address.getProperties();
                  String path = m_address.getPath();

                  String val = String.valueOf(value);
                  if (path != null) val = val + "/" + path;

                  JMXAddress address = new JMXAddress("jmx:" + val);
                  for (Iterator i = properties.entrySet().iterator(); i.hasNext();)
                  {
                     Map.Entry entry = (Map.Entry)i.next();
                     address.putProperty((String)entry.getKey(), entry.getValue());
                  }
                  m_address = address;
               }
            }
            else
            {
               if (m_address == null)
               {
                  // A dummy address, user should add more informations later
                  m_address = new JMXAddress("jmx:rmi://localhost");
               }
               m_address.putProperty((String)property, value);
            }
         }
         catch (MalformedJMXAddressException ignored)
         {
         }
      }
      else
      {
         throw new IllegalStateException("Cannot perform this operation while running");
      }
   }

   /**
    * Resets the JNDI properties set for this adaptor. <br>
    * This method can be called only if this adaptor is not running.
    *
    * @see #putJNDIProperty
    */
   public void clearJNDIProperties()
   {
      clearNamingProperties();
   }

   /**
    * @deprecated Replaced by {@link #clearJNDIProperties}
    */
   public void clearNamingProperties()
   {
      if (!isRunning())
      {
         try
         {
            if (m_address != null)
            {
               m_address = new JMXAddress(m_address.getProtocol(), m_address.getHost(), m_address.getPort(), m_address.getPath());
            }
         }
         catch (MalformedJMXAddressException ignored)
         {
         }
      }
      else
      {
         throw new IllegalStateException("Cannot perform this operation while running");
      }
   }

   /**
    * Returns the JNDI properties for this adaptor. <br>
    *
    * @see #putJNDIProperty
    */
   public Properties getJNDIProperties()
   {
      return getNamingProperties();
   }

   /**
    * @deprecated Replaced by {@link #getJNDIProperties}
    */
   public Properties getNamingProperties()
   {
      Properties p = new Properties();
      if (m_address != null)
      {
         Map properties = m_address.getProperties();
         if (properties.size() > 0)
         {
            StringBuffer buffer = new StringBuffer(m_address.getProtocol());
            buffer.append("://");
            buffer.append(m_address.getHost());
            int port = m_address.getPort();
            if (port > 0) buffer.append(":").append(port);
            p.put(Context.PROVIDER_URL, buffer.toString());
            for (Iterator i = properties.entrySet().iterator(); i.hasNext();)
            {
               Map.Entry entry = (Map.Entry)i.next();
               p.put(entry.getKey(), entry.getValue());
            }
         }
         else
         {
            // The address is there with only the provider URL, so probably the user wants to rely on defaults
            // Hence do nothing
         }
      }
      return p;
   }

   /**
    * Returns the JMXAddress for this adaptor.
    * Keep it private for now, as it is not strictly needed
    */
   private JMXAddress getJMXAddress()
   {
      return m_address == null ? null : (JMXAddress)m_address.clone();
   }

   /**
    * Sets the JMXAddress for this adaptor.
    * Keep it private for now, as it is not strictly needed
    */
   private void setJMXAddress(JMXAddress address)
   {
      m_address = (JMXAddress)address.clone();
   }

   /**
    * Returns the host name on which this adaptor is running
    */
   public String getHostName()
   {
      try
      {
         return InetAddress.getLocalHost().getHostName();
      }
      catch (UnknownHostException ignored)
      {
         return "localhost";
      }
   }

   /**
    * Returns the host address on which this adaptor is running
    */
   public String getHostAddress()
   {
      try
      {
         return InetAddress.getLocalHost().getHostAddress();
      }
      catch (UnknownHostException ignored)
      {
         return "127.0.0.1";
      }
   }

   //
   // Implementation of the remote interface
   //

   public InvocationResult invoke(Invocation invocation) throws Exception
   {
      Logger logger = getLogger();
      if (logger.isEnabledFor(Logger.TRACE)) logger.trace("Incoming RMI call, heading to interceptor chain");
      return m_headInterceptor.invoke(invocation);
   }

   //
   // Interceptor handling
   //

   /**
    * Adds an interceptor to this RMI adaptor
    */
   public void addInterceptor(Interceptor interceptor)
   {
      if (interceptor != null)
      {
         m_interceptors.add(interceptor);
      }
   }

   protected void installInterceptors()
   {
      MBeanServer server = getMBeanServer();
      if (server == null) throw new IllegalStateException("Target MBeanServer is not set. Either call setMBeanServer or register this MBean");

      List interceptors = getInterceptors();

      LoggerAdaptorInterceptor head = new LoggerAdaptorInterceptor();
      head.setMBeanServer(server);
      registerInterceptor(head);

      ContextClassLoaderAdaptorInterceptor neck = new ContextClassLoaderAdaptorInterceptor();
      head.setNext(neck);
      neck.setMBeanServer(server);
      registerInterceptor(neck);

      Interceptor previous = neck;
      for (int i = 0; i < interceptors.size(); ++i)
      {
         Interceptor next = (Interceptor)interceptors.get(i);
         previous.setNext(next);

         if (next instanceof AdaptorInterceptor)
         {
            ((AdaptorInterceptor)next).setMBeanServer(server);
         }

         registerInterceptor(next);
         previous = next;
      }

      InvokerAdaptorInterceptor last = new InvokerAdaptorInterceptor();
      previous.setNext(last);
      last.setMBeanServer(server);
      registerInterceptor(last);

      m_headInterceptor = head;
   }

   private void registerInterceptor(Interceptor interceptor)
   {
      ObjectName name = getObjectName();

      if (name == null)
      {
         // Means that this interceptor has not been registered, don't register the interceptors
         return;
      }

      Logger logger = getLogger();

      try
      {
         ObjectName interceptorObjectName = null;

         if (interceptor instanceof AdaptorInterceptor)
         {
            AdaptorInterceptor i = (AdaptorInterceptor)interceptor;
            interceptorObjectName = i.getObjectName();
         }

         if (interceptorObjectName == null)
         {
            // Deduct the interceptor name from the adaptor name

            if (logger.isEnabledFor(Logger.DEBUG)) logger.debug("Adaptor ObjectName: " + name);

            String domain = name.getDomain();
            Hashtable props = name.getKeyPropertyList();
            props.put("interceptor", interceptor.getType());
            interceptorObjectName = new ObjectName(domain, props);
         }

         if (logger.isEnabledFor(Logger.DEBUG)) logger.debug("Adaptor interceptor ObjectName: " + interceptorObjectName);
         m_server.registerMBean(interceptor, interceptorObjectName);
         m_interceptorNames.add(interceptorObjectName);
         if (logger.isEnabledFor(Logger.TRACE)) logger.trace("Adaptor interceptor registered successfully");
      }
      catch (Exception x)
      {
         if (logger.isEnabledFor(Logger.TRACE)) logger.trace("Cannot register interceptor", x);
      }
   }

   protected void uninstallInterceptors()
   {
      Logger logger = getLogger();

      for (int i = 0; i < m_interceptorNames.size(); ++i)
      {
         ObjectName name = (ObjectName)m_interceptorNames.get(i);
         if (logger.isEnabledFor(Logger.DEBUG)) logger.debug("Unregistering adaptor interceptor with ObjectName: " + name);

         try
         {
            m_server.unregisterMBean(name);
            if (logger.isEnabledFor(Logger.TRACE)) logger.trace("Adaptor interceptor unregistered successfully");
         }
         catch (Exception x)
         {
            if (logger.isEnabledFor(Logger.TRACE)) logger.trace("Cannot unregister interceptor", x);
         }
      }

      // Does not matter to uninstall also the interceptor chain: at this point no more RMI calls can
      // income, and the adaptor must be restarted, recreating the interceptor chain.
   }

   private List getInterceptors()
   {
      return m_interceptors;
   }

   //
   // Public methods
   //

   /**
    * Sets the target MBeanServer in case this adaptor is not registered with it
    */
   public void setMBeanServer(MBeanServer server)
   {
      m_server = server;
   }

   //
   // Protected methods for subclass usage
   //

   protected MBeanServer getMBeanServer()
   {
      return m_server;
   }

   protected ObjectName getObjectName()
   {
      return m_objectName;
   }

   protected void bind(Object adaptor) throws NamingException
   {
      Logger logger = getLogger();

      String jndiName = getJNDIName();
      if (logger.isEnabledFor(Logger.DEBUG)) logger.debug("Binding " + this + " in JNDI, name is: " + jndiName);
      if (jndiName == null) throw new NamingException("Invalid JNDI name");

      InitialContext context = null;
      Properties properties = getJNDIProperties();
      if (logger.isEnabledFor(Logger.DEBUG)) logger.debug("Binding " + this + " in JNDI, properties are: " + properties);
      if (properties.size() > 0)
         context = new InitialContext(properties);
      else
         context = new InitialContext();

      // Bind the remote object to the given name
      context.bind(jndiName, adaptor);
      if (logger.isEnabledFor(Logger.DEBUG)) logger.debug("Bound " + this + " in JNDI");
   }

   protected void unbind() throws NamingException
   {
      String jndiName = getJNDIName();
      if (jndiName == null)
      {
         throw new NamingException("Invalid JNDI name");
      }

      InitialContext context = null;
      Properties properties = getJNDIProperties();
      if (properties.size() != 0)
      {
         context = new InitialContext(properties);
      }
      else
      {
         context = new InitialContext();
      }
      // Unbind the given name
      context.unbind(jndiName);
   }

   private Logger getLogger()
   {
      return Log.getLogger(getClass().getName());
   }
}
