## By tommie, stefan@kann-nix.org
#
#  openurl.pl listens to every message that flows through your channels
#  and watches for www/ftp/mail addresses. Whenever it encounters one, the URL
#  will be stored in a list where it can be retrieved by /url <number>.
#  The list can be inspected by the command /urls -l, and for a better
#  orientation a reference mark is added to the URL in channel.
#
#  /set openurl_show_references ON/OFF
#	This controls whether [URL X] remarks should be displayed.
#
#  /set openurl_manipulate_message ON/OFF
#	This controls whether an additional message is generated or
#	the message containing the url is manipulated
#
#  /set openurl_max_urls
#	This setting specifies how many URLs should be saved. If there
#	appears a new one, the oldest will be deleted.
#  
#  /set openurl_browser_[ftp, http, mailto]
#	Here you can name your browser. Die command that is mentioned
#	here will be supplied with the URL as the first commandline argument.
#
#  /set openurl_save_context
#	This setting controls whether the whole message is saved or only the
#	included url
#
#  /set openurl_url_[ftp, http, mailto]
#	These options control what kind of urls are saved by openurl.pl.
#
# /JOIN #irssi.de
# Have Fun!
#
# History:
#
# 22.01.2002
# *First release to the public
#
# 23.01.2002
# *Optimized process_line (now using find_channel())
#
# 26.01.2002
# *Changed messagelevel to CLIENTCRAP
#
# 27.01.2002
# *Fixed highlight_url (prevent "coloured URLS)
#
# 28.01.2002
# *Tweaked mail-RegExp
# *"mailto:" is not added anymore
# *added headers
#
# 29.01.2002
# *Got back to the roots to fix some bugs
#
# 03.02.2002
# *tweaked overwriting of old urls
#
# 04.02.2002
# *fixed overwriting
#
# 15.02.2002
# *prepared data structures for future extensions
# *Implemented replacement of notes
#
# 16.02.2002
# *multiple notes are now correctly deleted
#
# 17.02.2002
# *fixed default boolean values
# 
# 21.02.2002
# *added -p (paste) option
#
# 27.02.2002
# *made removal of expired notes optional (buggy)
# 
# 01.03.2002
# *CHanged license to GPL
#
# 08.03.2002
# *some bugfixes
#
# 09.03.2002
# *Finally fixed the crash surrounding remove_line :)
#
# 10.03.2002
# *Some bugfixes


use strict;
use warnings;

use vars qw($VERSION %IRSSI);

$VERSION = "20020311";
%IRSSI = (
	authors     => "Stefan Tomanek",
	contact     => "stefan\@pico.ruhr.de",
	name        => "openurl",
	description => "Stores urls in a list and launches mail, web or ftp software",
	license     => "GPLv2",
	url         => "",
	changed	    => "$VERSION",
);


use Irssi;
use Irssi::TextUI;
use Irssi::Irc;

use vars qw(@urls $expire);
$expire=0;

sub highlight_url {
	my ($text) = @_;
	# Escape any "%"-ChaRS
	$text =~ s/%/%%/g;
	# Underline the URL
	my $url = fetch_url($text);
	$text =~ s/\Q$url/%U$url%U/;
	return($text);
}

sub del_notes {
	my (%url) = @_;
	my $view;
	my $target = $url{'target'};
	if (Irssi::channel_find($target)) {
		$view = Irssi::channel_find($target)->window()->view();
	} elsif (Irssi::query_find($target)) {
		$view = Irssi::query_find($target)->window()->view();
	}
	if ($view) {
		if (Irssi::settings_get_bool('openurl_remove_expired')) {
			#foreach (@{$url{'notes'}}) {
			foreach (@{$url{'notes'}}) {
				my $line = $view->get_bookmark($_);
				# Check if bookmark is still valid
				if ($line) {
					$view->remove_line($line);
					$view->redraw();
				}
				$view->set_bookmark($_, undef);
			}
			@{$url{'notes'}}=();
		}
	}
}
	
sub cmd_url {
	my ($arg, $server, $witem) = @_;
	my $paste_only=0;
	# List URLs if no other action is specified.
	unless ($arg) {
		$arg = "-l";
	}
	# Scan topics of channels we are in for any URLs.	
	if ($arg eq "-t") {
		my $oldurls = scalar(@urls);
		foreach my $channel (Irssi::channels()) {
			event_topic_changed($channel);
		}
		Irssi::print("%R>>%n ".(scalar(@urls)-$oldurls)." URLs added from channel topics", MSGLEVEL_CLIENTCRAP);
		return;
	}
	# Clear the list.
	if ($arg eq "-c") {
		while (scalar(@urls) > 0) {
			del_notes %{$urls[scalar(@urls)-1]};
			pop @urls;
		}
		Irssi::print("%R>>%n URLs cleared", MSGLEVEL_CLIENTCRAP);
		$expire=0;
		return;
	}
	if ($arg eq "-l") {
		Irssi::print("%R>>%n URLs:", MSGLEVEL_CLIENTCRAP);
		for (my $i = 0; $i < scalar(@urls); $i++) {
			my $num = $i+1;
			Irssi::print("%W[$num]=>%n <".$urls[$i]->{'target'}."> ".highlight_url($urls[$i]->{'text'}), MSGLEVEL_CRAP);
		}
		Irssi::print("%R>>%n End of URLs", MSGLEVEL_CLIENTCRAP);
		return;
	}
	$paste_only=1 if ($arg =~ /-p.*/);
	
	my @nums;
	foreach (split(/ /, $arg)) {
			push(@nums, $_) if ($_ =~ /[0-9].*/);
	}

	for my $num (@nums) {
		if ( int($num) ) {
			unless ($urls[$num-1]) {
				Irssi::print("%R>>%n No such message in URL buffer", MSGLEVEL_CLIENTCRAP);
			} else {
				if (not $paste_only) {
					open_browser(has_url($urls[$num-1]->{'text'}), fetch_url($urls[$num-1]->{'text'}));
				} else {
					if ($witem->{type} eq "CHANNEL" || $witem->{type} eq "QUERY") {
						$witem->command("/MSG ".$witem->{name}." ".fetch_url($urls[$num-1]->{'text'}));
					}
				}
			}
		} 
	}
}

sub extract {
	my ($text, $type) = @_;
	# Mail adresses require different handling, they can be 
	# processed either in "mailto" notation or without the leading
	# identifier.
	if ($type eq "mailto") {
		my @split_at=split(/\@/, $text);
		# The ":" removes any leading "mailto:" strings
		my @split_1=split(/[\ \:]/, $split_at[0]);
		my @split_2=split(/\ /, $split_at[1]);
		my $num_1=scalar(@split_1)-1;
		return($split_1[$num_1]."@".$split_2[0]); 
	} else {
		my @split1=split(/$type\:\/\//, $text);
		my @split2=split(/ /,$split1[1]);
		return("$type\:\/\/$split2[0]");
	}
}


sub fetch_url {
	my ($text) = @_;
	return(extract($text,has_url($text)));
}

sub has_url {
	my ($text) = @_;
	($text =~ /.*http:\/\/.*/) && return("http");
	($text =~ /.*ftp:\/\/.*/) && return("ftp");
	# since nobody uses url-notations with email, we look for @s
	($text =~ /\w+@\w+\.[a-z]{2,}/i) && return("mailto");
	return("");
}

sub open_browser {
	my ($type,$url) = @_;
	my $command = Irssi::settings_get_str("openurl_browser_$type");
	system("$command "."'".$url."'");
}

sub add_url {
	my ($text, $target) = @_;
	my $position=-1;
	for (my $i = 0; $i < scalar(@urls); $i++) {
		if ("$text" eq $urls[$i]->{'text'}) {
			return( (0,$i+1) );
		}
	}
	if ( (has_url($text)) and (not Irssi::settings_get_bool("openurl_proto_".has_url($text))) ) {
		return( (-1,-1) );
	}
	# Check if we reached the last passoble URL slot...
	my $max_urls = Irssi::settings_get_int("openurl_max_urls");
	
	del_notes %{$urls[$expire]} if ($urls[$expire]);
	my %url;
	$url{'text'} = $text;
	$url{'notes'} = ();
	$url{'target'} = $target;
	$urls[$expire] = \%url;
	$position = $expire+1;
	if ($expire == $max_urls-1) { $expire = 0; } else { $expire++; }
	return( (1,$position) );
}

sub process_line {
	my ($server, $target, $line) = @_;
	my ($success, $position);
	# Check if there is any URL in the line
	if (has_url($line)) {
		# Do we only want to store the URL or the entire line=
		if (Irssi::settings_get_bool("openurl_save_context")) {
			($success, $position) = add_url($line, $target);
		} else {
			($success, $position) = add_url(fetch_url($line), $target);
		}
		if (Irssi::settings_get_bool("openurl_show_references")) {
			# Locate the Window-Item
			my $chan;
			if ($server->channel_find($target)) {
				$chan = $server->channel_find($target);
			} else {
				$chan = $server->query_find($target);
			}
			if ($success==1) {
				add_note($chan, $position);
			} elsif ($success==0) {
				add_note($chan, $position);
			} elsif ($success==-1) {
				# Just ignore it :);
			}
		}
	}
	return($success);
}

sub add_note {
	my ($channel, $position) = @_;
	$channel->print("%R>>%n URL $position", MSGLEVEL_CLIENTCRAP);
	my $line = last_line($channel);
	my $time = time();
	$channel->window()->view()->set_bookmark("url_".$position."-".$time, $line);
	push @{$urls[$position-1]->{'notes'}}, "url_".$position."-".$time;
}

sub last_line {
	my ($channel) = @_;
	my $window = $channel->window();
	my $view = $window->view();
	#my $buffer = $view->{buffer};
	my $line = $view->{bottom_startline};
	
	while ($line->next()) {
		$line = $line->next();
	}
	return $line;
}
sub event_public_message {
	my ($server, $text, $nick, $address, $target) = @_;
	process_line($server, $target, $text);
}

sub event_private_message {
	my ($server, $text, $nick, $address) = @_;
	process_line($server, $nick, $text);
}

sub event_topic_changed {
	my ($channel) = @_;
	process_line($channel->{server}, $channel->{name}, $channel->{topic});
}

Irssi::command_bind("urls", "cmd_url");

Irssi::signal_add_last("message public", "event_public_message");
Irssi::signal_add_last("message private", "event_private_message");
Irssi::signal_add_last("channel topic changed", "event_topic_changed");

Irssi::settings_add_str("misc", "openurl_browser_http", "screen w3m");
Irssi::settings_add_str("misc", "openurl_browser_ftp", "screen ncftp");
Irssi::settings_add_str("misc", "openurl_browser_mailto", "screen mutt");

Irssi::settings_add_str("misc", "openurl_max_urls", -1);
Irssi::settings_add_bool("misc", "openurl_show_references", 1);
Irssi::settings_add_bool("misc", "openurl_save_context", 1);
Irssi::settings_add_bool("misc", "openurl_remove_expired", 1);

Irssi::settings_add_bool("misc", "openurl_proto_http", 1);
Irssi::settings_add_bool("misc", "openurl_proto_ftp", 1);
Irssi::settings_add_bool("misc", "openurl_proto_mailto", 1);

