/* MutableGraphPanel.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.gred;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.grinvin.graphs.Annotation;
import org.grinvin.graphs.AnnotationModel;
import org.grinvin.graphs.DefaultEmbedding;
import org.grinvin.graphs.Edge;
import org.grinvin.graphs.Element;
import org.grinvin.graphs.Embedding;
import org.grinvin.graphs.EmbeddingModel;
import org.grinvin.graphs.EmbeddingView;
import org.grinvin.graphs.MutableAnnotationModel;
import org.grinvin.graphs.MutableEmbeddingModel;
import org.grinvin.gred.undoable.UndoableGraph;
import org.grinvin.graphs.Vertex;
import org.grinvin.gred.undoable.AddElements;
import org.grinvin.gred.undoable.ChangeCoordinates;
import org.grinvin.gred.undoable.InternationalizedUndoableChange;
import org.grinvin.gred.undoable.RemoveElements;
import org.grinvin.graphs.render.Renderer;

/**
 * Graph panel which is backed by a 2-dimensional mutable embedding model.
 * Extends {@link GraphPanel} with the ability to add vertices and change
 * vertex coordinates.
 */
public class MutableGraphPanel extends GraphPanel {
    
    //
    private boolean dirty;
    
    //
    public boolean isDirty() {
        return dirty;
    }
    
    //
    public void setDirty(boolean dirty) {
        this.dirty = dirty;
    }
    
    // embedding which is wrapped by the dynamic embedding
    
    /**
     * Create a new graph panel with given peer. The panel is given a preferred size
     * large enough to display the coordinate range (-1.1,-1.1)-(1.1,1.1).
     * @param embedding Embedding displayed in this panel. The peer of this embedding
     * should be of type MutableGraphModel.
     * @param renderer Renderer for this panel.
     * @param context Graph context for this panel.
     * @param scale Number of pixels corresponding to a unit length in the embedding.
     */
    public MutableGraphPanel(MutableEmbeddingModel embedding, MutableAnnotationModel annotation,
            Renderer renderer, GraphContext context, double scale) {
        super(embedding, annotation, renderer, context, scale);
        this.snapToGuides = false;
        this.dirty = false;
    }
    
    // overrides GraphPanel
    @Override public void setGraph(EmbeddingModel originalEmbedding, AnnotationModel annotation) {
        if (originalEmbedding instanceof MutableEmbeddingModel) {
            super.setGraph(originalEmbedding, annotation);
        } else
            throw new IllegalArgumentException("Embedding model must be mutable");
    }
    
    // overrides GraphPanel
    @Override public void close() {
        super.close();
    }
    
    /**
     * Returns the (mutable) graph displayed by this panel.
     */
    @Override public UndoableGraph getGraph() {
        return (UndoableGraph)this.graph;
    }
    
    /**
     * Returns the (mutable) embedding used by this panel.
     */
    @Override public Embedding getEmbedding() {
        return (Embedding)this.embedding;
    }
    
    /**
     * Returns the (mutable) annotation used by this panel.
     */
    @Override public Annotation getAnnotation() {
        return (Annotation)this.annotation;
    }
    
    /**
     * Removes the current graph
     */
    public void clear(){
        List<Element> list = new ArrayList<Element>();
        for(Iterator<Edge> iterator = graph.edgeIterator(); iterator.hasNext(); ) {
            Edge e = iterator.next();
            list.add(e);
            iterator.remove();
        }
        for(Iterator<Vertex> iterator = graph.vertexIterator(); iterator.hasNext(); ) {
            Vertex v = iterator.next();
            list.add(v);
            iterator.remove();
        }
        
        if(list.size()>0){
            undoManager.add(new RemoveElements(getGraph(), list), true);
        }
    }
    
    
    /* ============================================================
     * GUIDES
     * ============================================================ */
    
    //
    private boolean snapToGuides;
    
    /**
     * Indicates whether new vertices added should be 'snapped'
     * to the guides.
     */
    public void setSnapToGuides(boolean snapToGuides) {
        this.snapToGuides = snapToGuides;
    }
    
    
    /* ============================================================
     * ADD ELEMENTS
     * ============================================================ */
    
    
    /**
     * Add a new vertex with given mouse coordinates and annotations.
     * Snaps to guides if currently enabled.
     */
    private Vertex addNewVertex(final int mouseX, final int mouseY, final Object annotation) {
        Vertex vertex = getGraph().addNewVertex();
        getAnnotation().setAnnotation(vertex, annotation);
        double[] coords = new double[] { (mouseX - getWidth()/2)/scale,
        (getHeight()/2 - mouseY)/scale };
        if (snapToGuides)
            guides.snap(coords);
        getEmbedding().setCoordinates(vertex, coords);
        setDirty(true);
        return vertex;
    }
    
    /**
     * Add a new vertex at the given mouse coordinates
     * using the given annotation.
     * @param leader Is this command the first of an undo/redo group?
     * @return the newly created vertex
     */
    public Vertex addNewVertex(int mouseX, int mouseY, Object annotation, boolean leader) {
        // execute action
        Vertex vertex = addNewVertex(mouseX, mouseY, annotation);
        
        // register with undoManager
        undoManager.add(new AddElements(getGraph(), vertex ), leader);
        context.setRollOver(vertex);
        setDirty(true);
        return vertex;
    }
    
    /**
     * Split an edge and optionally join the new vertex to the anchor.
     */
    public void splitEdge(Edge edge, boolean joinToAnchor,
            int mouseX, int mouseY, Object annotation, boolean leader) {
        
        joinToAnchor &= (anchor instanceof Vertex);
        
        // execute action
        Vertex vertex = addNewVertex(mouseX, mouseY, annotation);
        
        ArrayList<Element> elements = new ArrayList<Element> ();
        elements.add(vertex);
        
        context.setRollOver(vertex);
        Vertex first = edge.getFirstEndpoint();
        Vertex second = edge.getSecondEndpoint();
        getGraph().remove(edge);
        elements.add(getGraph().addNewEdge(first, vertex));
        elements.add(getGraph().addNewEdge(vertex, second));
        
        if (joinToAnchor) {
            elements.add(getGraph().addNewEdge((Vertex)anchor, vertex));
        }
        
        // register with undoManager
        if (leader) {
            undoManager.add
                    (new InternationalizedUndoableChange(joinToAnchor ? "SplitAndJoin" : "Split"),
                    true);
        }
        undoManager.add(new RemoveElements(getGraph(), edge), false);
        undoManager.add(new AddElements(getGraph(), elements), false);
        setDirty(true);
    }
    
    /**
     * CHANGES IN EMBEDDINGS
     */
    
    //
    private Embedding savedEmbedding = null; // lazily created
    
    /**
     * Return the embedding saved by {@link #initiateEmbeddingChange}.
     */
    public EmbeddingView getSavedEmbedding() {
        return this.savedEmbedding;
    }
    
    /**
     * Initiate an upcoming change to an embedding. Saves a copy of the current
     * embedding. This copy can be retrieved later by means of {@link #getSavedEmbedding}.
     */
    public void initiateEmbeddingChange() {
        savedEmbedding = new DefaultEmbedding(embedding);
    }
    
    /**
     * Finalize a change in embedding. Hands the change over to the undo manager.
     * @param key Resource key for the caption of the undo/redo button
     */
    public void finalizeEmbeddingChange(String key) {
        
        // create collection of movements
        ChangeCoordinates cc = new ChangeCoordinates(getEmbedding(), key);
        for (Vertex v : graph.vertices()) {
            cc.addMovement(v, savedEmbedding.getCoordinates(v), embedding.getCoordinates(v) );
        }
        
        // register with undo manager
        undoManager.add(cc, true);
    }
    
    /**
     * Undo the latest embedding change, by reverting to the coordinates
     * of the embedding saved by {@link #initiateEmbeddingChange}.
     */
    public void undoEmbeddingChange() {
        getEmbedding().copy(savedEmbedding);
    }
    
    /**
     * Move the current selection by the given number of
     * pixels (in mouse coordinates)
     */
    public void moveSelection(int diffx, int diffy) {
        double dx = diffx / scale;
        double dy = -diffy / scale;
        
        for (Vertex v : graph.vertices()) {
            if (context.isSelected(v)) {
                double[] c = savedEmbedding.getCoordinates(v);
                c[0] += dx;
                c[1] += dy;
                getEmbedding().setCoordinates(v, c);
            }
        }
        setDirty(true);
    }
    
    /**
     * Snap all the vertices to the current guides.
     */
    public void snapAllToGuides() {
        if (guides == null)
            return; // do nothing
        
        initiateEmbeddingChange();
        
        // snap all
        for (Vertex v : graph.vertices()) {
            double[] c = savedEmbedding.getCoordinates(v);
            guides.snap(c);
            getEmbedding().setCoordinates(v, c);
        }
        
        finalizeEmbeddingChange("SnapAll");
        setDirty(true);
    }
        
    /*============================================================
     * MANIPULATE THE SELECTION
     *============================================================*/
    
    
    /**
     * Delete the current selection.
     */
    public void deleteSelection() {
        // perform action
        Collection<Element> elements
                = context.getSelection(new HashSet<Element> ());
        // remove zombies
        for (Iterator<Element> iterator = elements.iterator();
        iterator.hasNext(); ) {
            Element el = iterator.next();
            if (el instanceof Vertex && !graph.contains((Vertex)el))
                iterator.remove();
            else if (el instanceof Edge && !graph.contains((Edge)el))
                iterator.remove();
        }
        
        if (elements.isEmpty())
            return;
        
        context.setRollOver(null);
        
        // compute extra edges which will be removed with these elements
        // TODO: duplicates code executed by DefaultGraph.remove,
        // probably best to remove restore/remove etc from Graph
        // into DefaultGraphModel
        
        // remove (and register) everything, edges first
        List<Element> list = new ArrayList<Element>();
        for (Iterator<Edge> iterator = graph.edgeIterator(); iterator.hasNext(); ) {
            Edge edge = iterator.next();
            if (elements.contains(edge) ||
                    elements.contains(edge.getFirstEndpoint()) ||
                    elements.contains(edge.getSecondEndpoint())) {
                list.add(edge);
                iterator.remove();
            }
        }
        for (Element el: elements) {
            if (el instanceof Vertex) {
                list.add(el);
                getGraph().remove((Vertex)el);
            }
        }
        
        // register with undo manager
        undoManager.add(new RemoveElements(getGraph(), list), true);
        setDirty(true);
    }
    
    /**
     * Snap the current selection to the current guides.
     */
    public void snapSelectionToGuides() {
        if (context.isSelectionEmpty() || guides == null)
            return; // do nothing
        
        initiateEmbeddingChange();
        
        // snap selection
        for (Vertex v : graph.vertices()) {
            if (context.isSelected(v)) {
                double[] c = savedEmbedding.getCoordinates(v);
                guides.snap(c);
                getEmbedding().setCoordinates(v, c);
            }
        }
        
        finalizeEmbeddingChange("SnapSelection");
        setDirty(true);
    }
    
    /**
     * Merges the current selected vertices.
     */
    public void mergeSelection() {
        if (context.isSelectionEmpty())
            return; // do nothing
        
        List<Vertex> selectedVertices = new ArrayList<Vertex>();
        Set<Vertex> adjacentVertices = new HashSet<Vertex>();
        double[] coords = new double[2];
        int nrOfVerticesInSelection = 0;
        
        for (Vertex v : graph.vertices()) {
            if (context.isSelected(v)) {
                coords[0] += embedding.getCoordinates(v)[0];
                coords[1] += embedding.getCoordinates(v)[1];
                nrOfVerticesInSelection++;
                selectedVertices.add(v);
                for (Vertex v2 : graph.vertices())
                    if(graph.areAdjacent(v,v2) && !context.isSelected(v2))
                        adjacentVertices.add(v2);
            }
        }
        
        if(nrOfVerticesInSelection<2)
            return; // do nothing
        
        coords[0] /= nrOfVerticesInSelection;
        coords[1] /= nrOfVerticesInSelection;
        Vertex vertex = getGraph().addNewVertex();
        if (snapToGuides)
            guides.snap(coords);
        getEmbedding().setCoordinates(vertex, coords);
        
        List<Element> edges = new ArrayList<Element>();
        for(Vertex v : adjacentVertices)
            edges.add(getGraph().addNewEdge(vertex, v));
        
        // remove selected vertices and their edges
        List<Element> list = new ArrayList<Element>();
        for (Iterator<Edge> iterator = graph.edgeIterator(); iterator.hasNext(); ) {
            Edge edge = iterator.next();
            if (selectedVertices.contains(edge.getFirstEndpoint()) || selectedVertices.contains(edge.getSecondEndpoint())) {
                list.add(edge);
                iterator.remove();
            }
        }
        for (Element el: selectedVertices) {
            list.add(el);
            getGraph().remove((Vertex)el);
        }
        
        undoManager.add(new InternationalizedUndoableChange("MergeSelection"), true);
        undoManager.add(new AddElements(getGraph(), vertex), false);
        undoManager.add(new AddElements(getGraph(), edges), false);
        undoManager.add(new RemoveElements(getGraph(), list), false);
        setDirty(true);
    }
    
    /**
     * Splits the current selected vertices.
     */
    public void splitSelection(){
        if (context.isSelectionEmpty())
            return; // do nothing
        
        List<Vertex> selectedVertices = new ArrayList<Vertex>();
        for (Vertex v : graph.vertices())
            if (context.isSelected(v))
                selectedVertices.add(v);
        if(selectedVertices.size()==0)
            return; //do nothing
        
        undoManager.add(new InternationalizedUndoableChange("SplitSelection"), true);
        
        for(Vertex v : selectedVertices){
            double[] vCoords = embedding.getCoordinates(v);
            Vertex[] vertices = new Vertex[graph.getNumberOfVertices()];
            int index = 0;
            for(Vertex v2 : graph.vertices())
                vertices[index++] = v2;
            for(Vertex vertex : vertices){
                if(graph.areAdjacent(v,vertex)){
                    Vertex newVertex = getGraph().addNewVertex();
                    double[] coords = new double[2];
                    coords[0] = vCoords[0] + (embedding.getCoordinates(vertex)[0]-vCoords[0])/5;
                    coords[1] = vCoords[1] + (embedding.getCoordinates(vertex)[1]-vCoords[1])/5;
                    getEmbedding().setCoordinates(newVertex, coords);
                    Edge e = graph.getEdge(v, vertex) != null ? graph.getEdge(v, vertex) : graph.getEdge(vertex, v);
                    getGraph().remove(e);
                    undoManager.add(new RemoveElements(getGraph(), e), false);
                    undoManager.add(new AddElements(getGraph(), newVertex), false);
                    undoManager.add(new AddElements(getGraph(), getGraph().addNewEdge(newVertex, vertex)), false);
                }
            }
            getGraph().remove(v);
            undoManager.add(new RemoveElements(getGraph(), v), false);
        }
        setDirty(true);
    }
    
    /**
     * Connects the current selected vertices (i.e. they become a clique).
     */
    public void connectSelection(){
        if (context.isSelectionEmpty())
            return; // do nothing
        
        List<Vertex> selectedVertices = new ArrayList<Vertex>();
        List<Element> newEdges = new ArrayList<Element>();
        for (Vertex v : graph.vertices())
            if (context.isSelected(v))
                selectedVertices.add(v);
        for(int i=0; i<selectedVertices.size()-1; i++)
            for(int j=i+1; j<selectedVertices.size(); j++)
                if(!graph.areAdjacent(selectedVertices.get(i),selectedVertices.get(j)))
                    newEdges.add(getGraph().addNewEdge(selectedVertices.get(i),selectedVertices.get(j)));
        
        undoManager.add(new AddElements(getGraph(), newEdges), true);
        setDirty(true);
    }
    
    /**
     * Replaces the current graph with the graph induced by the selected vertices.
     */
    public void vertexInducedSubGraph(){
        List<Element> notSelectedVertices = new ArrayList<Element>();
        for (Vertex v : graph.vertices())
            if (!context.isSelected(v))
                notSelectedVertices.add(v);
        
        if(notSelectedVertices.size()==0)
            return; //do nothing
        
        List<Element> list = new ArrayList<Element>();
        for (Iterator<Edge> iterator = graph.edgeIterator(); iterator.hasNext(); ) {
            Edge edge = iterator.next();
            if (notSelectedVertices.contains(edge.getFirstEndpoint()) || notSelectedVertices.contains(edge.getSecondEndpoint())) {
                list.add(edge);
                iterator.remove();
            }
        }
        for (Element el: notSelectedVertices) {
            list.add(el);
            getGraph().remove((Vertex)el);
        }
        
        // register with undo manager
        undoManager.add(new RemoveElements(getGraph(), list), true);
        setDirty(true);
    }
    
    /**
     * Replaces the current graph with the graph induced by the selected edges.
     */
    public void edgeInducedSubGraph(){
        Collection<Vertex> remainingVertices = new HashSet<Vertex>();        
        List<Element> list = new ArrayList<Element>();
        for (Iterator<Edge> iterator = graph.edgeIterator(); iterator.hasNext(); ) {
            Edge e = iterator.next();
            if (context.isSelected(e)) {
                remainingVertices.add(e.getFirstEndpoint());
                remainingVertices.add(e.getSecondEndpoint());
            } else {
                list.add(e);
                iterator.remove();
            }
        }
        
        if(list.size()==0)
            return; //we can stop here
        
        for (Iterator<Vertex> iterator = graph.vertexIterator(); iterator.hasNext(); ) {
            Vertex v = iterator.next();
            if(!remainingVertices.contains(v)){
                list.add(v);
                iterator.remove();
            }
        }
        
        // register with undo manager
        undoManager.add(new RemoveElements(getGraph(), list), true);
        setDirty(true);
    }
    
    /*============================================================
     * GRAPH OPERATIONS
     *============================================================*/
    
    
    /**
     * Replaces the current graph with its complement.
     */
    public void complement(){
        int n = graph.getNumberOfVertices();
        if(n<2)
            return;
        Vertex[] vertices = new Vertex[n];
        int index = 0;
        for(Vertex v : graph.vertices())
            vertices[index++]=v;
        List<Element> removedEdges = new ArrayList<Element>();
        List<Element> addedEdges = new ArrayList<Element>();
        for(int i = 0; i<n-1; i++) {
            for(int j = i+1; j<n;j++) {
                if(graph.areAdjacent(vertices[i], vertices[j])){
                    Edge e = graph.getEdge(vertices[i], vertices[j]) != null ? graph.getEdge(vertices[i], vertices[j]) : graph.getEdge(vertices[j], vertices[i]);
                    removedEdges.add(e);
                    getGraph().remove(e);
                } else {
                    addedEdges.add(getGraph().addNewEdge(vertices[i],vertices[j]));
                }
            }
        }
        undoManager.add(new InternationalizedUndoableChange("Complement"), true);
        undoManager.add(new AddElements(getGraph(), addedEdges), false);
        undoManager.add(new RemoveElements(getGraph(), removedEdges), false);
        setDirty(true);
    }
    
    /**
     * Replaces the current graph with its closure
     */
    public void closure(){
        //first look up vertex degrees
        int n = graph.getNumberOfVertices();
        Vertex[] vertices = new Vertex[n];
        int index = 0;
        for(Vertex v : graph.vertices())
            vertices[index++]=v;
        int[] degree = new int[n];
        for(int i=0; i<n-1; i++)
            for(int j=i+1; j<n; j++)
                if(graph.areAdjacent(vertices[i], vertices[j])){
                    degree[i]++;
                    degree[j]++;
                }
        List<Element> addedEdges = new ArrayList<Element>();
        int previousSize=-1;
        while(addedEdges.size()!=previousSize){
            previousSize = addedEdges.size();
            for(int i=0; i<n-1; i++)
                for(int j=i+1; j<n; j++)
                    if(!graph.areAdjacent(vertices[i], vertices[j]) && degree[i]+degree[j]>=n){
                        addedEdges.add(getGraph().addNewEdge(vertices[i], vertices[j]));
                        degree[i]++;
                        degree[j]++;
                    }
        }
        if(addedEdges.size()>0){
            undoManager.add(new AddElements(getGraph(), addedEdges), true);
            setDirty(true);
        }
    }
    
    /**
     * Replaces the current graph with its line graph
     */
    public void lineGraph(){
        List<Element> oldEdges = new ArrayList<Element>();
        List<Element> oldVertices = new ArrayList<Element>();
        List<Element> addedEdges = new ArrayList<Element>();
        List<Element> addedVertices = new ArrayList<Element>();
        for(Vertex v : graph.vertices())
            oldVertices.add(v);
        int newOrder = graph.getNumberOfEdges();
        Vertex[] newVertices = new Vertex[newOrder];
        int index = 0;
        
        //place new vertices
        for(Edge e : graph.edges()){
            oldEdges.add(index, e);
            newVertices[index]=getGraph().addNewVertex();
            addedVertices.add(newVertices[index]);
            double[] coords = new double[2];
            coords[0] = (embedding.getCoordinates(e.getFirstEndpoint())[0] + embedding.getCoordinates(e.getSecondEndpoint())[0])/2;
            coords[1] = (embedding.getCoordinates(e.getFirstEndpoint())[1] + embedding.getCoordinates(e.getSecondEndpoint())[1])/2;
            getEmbedding().setCoordinates(newVertices[index], coords);
            index++;
        }
        
        //connect new vertices
        for(int i=0; i<newOrder-1; i++)
            for(int j=i+1; j<newOrder; j++)
                if(edgesIntersect((Edge)oldEdges.get(i), (Edge)oldEdges.get(j)))
                    addedEdges.add(getGraph().addNewEdge(newVertices[i], newVertices[j]));
        
        //remove old graph
        for(Element e : oldEdges)
            getGraph().remove((Edge)e);
        for(Element e : oldVertices)
            getGraph().remove((Vertex)e);
        
        undoManager.add(new InternationalizedUndoableChange("LineGraph"), true);
        undoManager.add(new AddElements(getGraph(), addedVertices), false);
        undoManager.add(new AddElements(getGraph(), addedEdges), false);
        undoManager.add(new RemoveElements(getGraph(), oldEdges), false);
        undoManager.add(new RemoveElements(getGraph(), oldVertices), false);
        setDirty(true);
    }
    
    //for use in lineGraph()
    private boolean edgesIntersect(Edge e1, Edge e2){
        return (e1.getFirstEndpoint().equals(e2.getFirstEndpoint()) || e1.getFirstEndpoint().equals(e2.getSecondEndpoint()) ||
                e1.getSecondEndpoint().equals(e2.getFirstEndpoint()) || e1.getSecondEndpoint().equals(e2.getSecondEndpoint()));
    }
}
