#ifndef MX_H
#define MX_H

/* 
  mx -- Marc's eXtension modules for Python: basic macros

  This file is only meant to be included by the extension modules.
  DO NOT include it in the extension module's header file, since it
  will definitely cause troubles then.

  To enable debugging ceratin things, define one of these before
  including this file:

  MAL_REF_DEBUG -- debug reference counts (Py_MY_xxx) [this file]
  MAL_DEBUG     -- enable debug output (DPRINTF) [mxstdlib.h]
  MAL_MEM_DEBUG -- enable malloc output (new,cnew,free,...) [mxstdlib.h]

  (c) Marc-Andre Lemburg; All Rights Reserved

  Additions from other sources:
  -----------------------------
  * Includes some additional macros from Python 1.5
  * Thanks to Christian Tismer for helping in porting this stuff
    to Windows; the import/export macros are based on his suggestions.
*/

/* Include the generic mx header file */
#include "mxh.h"

/* Include all Python symbols & definitions */
#include "Python.h"

/* Include other standard stuff */
#include "mxstdlib.h"

/* Include backward compatibility stuff */
#include "mxpyapi.h"

/* --- Declare macros ----------------------------------------------------- */

#define Py_NONE (Py_INCREF(Py_None),Py_None)

#ifdef MAL_REF_DEBUG
# define printref(x) printf("* refcount for "#x" = %i\n",(long) x->ob_refcnt);
#else
# define printref(x)
#endif

/*** Simple error catching ***/

#define Py_Do(x) {if (!(x)) goto onError;}
#define Py_ReturnOnError(errortype,errorstr) {PyErr_SetString(errortype,errorstr);return NULL;}

#define Py_Assert(x,errortype,errorstr) {if (!(x)) {PyErr_SetString(errortype,errorstr);goto onError;}}
#define Py_AssertWithArg(x,errortype,errorstr,a1) {if (!(x)) {PyErr_Format(errortype,errorstr,a1);goto onError;}}
#define Py_AssertWith2Args(x,errortype,errorstr,a1,a2) {if (!(x)) {PyErr_Format(errortype,errorstr,a1,a2);goto onError;}}
#define Py_AssertWith3Args(x,errortype,errorstr,a1,a2,a3) {if (!(x)) {PyErr_Format(errortype,errorstr,a1,a2,a3);goto onError;}}

#define Py_Error(errortype,errorstr) {PyErr_SetString(errortype,errorstr);goto onError;}
#define Py_ErrorWithArg(errortype,errorstr,a1) {PyErr_Format(errortype,errorstr,a1);goto onError;}
#define Py_ErrorWith2Args(errortype,errorstr,a1,a2) {PyErr_Format(errortype,errorstr,a1,a2);goto onError;}
#define Py_ErrorWith3Args(errortype,errorstr,a1,a2,a3) {PyErr_Format(errortype,errorstr,a1,a2,a3);goto onError;}

/*** Reference counting ***/

#ifdef MAL_REF_DEBUG
# define Py_MY_INCREF(x) Py_INCREF(x);printf("* refcount for "#x" = %i (after INCREF at %s:%i)\n",(long) x->ob_refcnt, __FILE__,__LINE__);
# define Py_MY_DECREF(x) Py_DECREF(x);printf("* refcount for "#x" = %i (after DECREF at %s:%i)\n",(long) x->ob_refcnt, __FILE__,__LINE__);
# define Py_MY_DELETE(x) if ((long) x->ob_refcnt > 1) printf("* Warning: refcount for "#x" = %i (NOT deleted in %s:%i)\n",(long) x->ob_refcnt-1, __FILE__,__LINE__);Py_DECREF(x);
# define Py_MY_REFCOUNT(x) printf("* refcount for "#x" = %i (REFCOUNT at %s:%i)\n",(long) x->ob_refcnt, __FILE__,__LINE__);
#else
# define Py_MY_INCREF(x) Py_INCREF(x);
# define Py_MY_DECREF(x) Py_DECREF(x);
# define Py_MY_DELETE(x) Py_DECREF(x);
# define Py_MY_REFCOUNT(x)
#endif

#define Py_PRINT_REFCOUNT(x) printf("* refcount for "#x" = %i (REFCOUNT at %s:%i)\n",(long) x->ob_refcnt, __FILE__,__LINE__);
#define Py_DEC_REF(x) Py_XDECREF(x); x=0; /* doing this once too often doesn't hurt */

/*** Argument parsing ***/

/* no arguments expected; also use Py_MethodListEntryNoArgs() for this
   kind of fct */
#define Py_NoArgsCheck() {if (!PyArg_NoArgs(args)) goto onError;}
/* for functions with old style args (Py_MethodListEntrySingleArg) */
#define Py_GetArgObject(a) {a = args; if (!a) {PyErr_SetString(PyExc_TypeError,"function/method requires an argument"); goto onError;}}
#define Py_GetSingleArg(format,a1) {if (!PyArg_Parse(args,format,&a1)) goto onError;}
/* for functions with new style args: */
#define Py_GetArg(format,a1) {if (!PyArg_ParseTuple(args,format,&a1)) goto onError;}
#define Py_Get2Args(format,a1,a2) {if (!PyArg_ParseTuple(args,format,&a1,&a2)) goto onError;}
#define Py_Get3Args(format,a1,a2,a3) {if (!PyArg_ParseTuple(args,format,&a1,&a2,&a3)) goto onError;}
#define Py_Get4Args(format,a1,a2,a3,a4) {if (!PyArg_ParseTuple(args,format,&a1,&a2,&a3,&a4)) goto onError;}
#define Py_Get5Args(format,a1,a2,a3,a4,a5) {if (!PyArg_ParseTuple(args,format,&a1,&a2,&a3,&a4,&a5)) goto onError;}
#define Py_Get6Args(format,a1,a2,a3,a4,a5,a6) {if (!PyArg_ParseTuple(args,format,&a1,&a2,&a3,&a4,&a5,&a6)) goto onError;}
#define Py_Get7Args(format,a1,a2,a3,a4,a5,a6,a7) {if (!PyArg_ParseTuple(args,format,&a1,&a2,&a3,&a4,&a5,&a6,&a7)) goto onError;}
#define Py_Get8Args(format,a1,a2,a3,a4,a5,a6,a7,a8) {if (!PyArg_ParseTuple(args,format,&a1,&a2,&a3,&a4,&a5,&a6,&a7,&a8)) goto onError;}
/* for functions with keywords -- the first macro parameter must be
   the keywords array given as e.g. {"first","second","third",NULL}
   with an entry for every argument (in the correct order). */
#define Py_KeywordGetArg(keywords,format,a1) {if (!PyArg_ParseTupleAndKeywords(args,kw,format,keywords,&a1)) goto onError;}
#define Py_KeywordGet2Args(keywords,format,a1,a2) {if (!PyArg_ParseTupleAndKeywords(args,kw,format,keywords,&a1,&a2)) goto onError;}
#define Py_KeywordGet3Args(keywords,format,a1,a2,a3) {if (!PyArg_ParseTupleAndKeywords(args,kw,format,keywords,&a1,&a2,&a3)) goto onError;}
#define Py_KeywordGet4Args(keywords,format,a1,a2,a3,a4) {if (!PyArg_ParseTupleAndKeywords(args,kw,format,keywords,&a1,&a2,&a3,&a4)) goto onError;}
#define Py_KeywordGet5Args(keywords,format,a1,a2,a3,a4,a5) {if (!PyArg_ParseTupleAndKeywords(args,kw,format,keywords,&a1,&a2,&a3,&a4,&a5)) goto onError;}
#define Py_KeywordGet6Args(keywords,format,a1,a2,a3,a4,a5,a6) {if (!PyArg_ParseTupleAndKeywords(args,kw,format,keywords,&a1,&a2,&a3,&a4,&a5,&a6)) goto onError;}
#define Py_KeywordGet7Args(keywords,format,a1,a2,a3,a4,a5,a6,a7) {if (!PyArg_ParseTupleAndKeywords(args,kw,format,keywords,&a1,&a2,&a3,&a4,&a5,&a6,&a7)) goto onError;}
#define Py_KeywordGet8Args(keywords,format,a1,a2,a3,a4,a5,a6,a7,a8) {if (!PyArg_ParseTupleAndKeywords(args,kw,format,keywords,&a1,&a2,&a3,&a4,&a5,&a6,&a7,&a8)) goto onError;}

/*** Returning values ***/

/* XXX don't always work: every time you have an 'O' in the BuildValue format
       string, you need to DECREF the variable *after* the tuple has been
       built !!!
*/

#define Py_ReturnNone() {Py_MY_INCREF(Py_None);return Py_None;}
#define Py_Return(format,x) return Py_BuildValue(format,x);
#define Py_Return2(format,x,y) return Py_BuildValue(format,x,y);
#define Py_Return3(format,x,y,z) return Py_BuildValue(format,x,y,z);

/* Build values */

#define Py_BuildNone() Py_NONE
#define Py_Build(format,x) Py_BuildValue(format,x)
#define Py_Build2(format,x,y) Py_BuildValue(format,x,y)
#define Py_Build3(format,x,y,z) Py_BuildValue(format,x,y,z)

/*** Standard C defs for local functions/methods ***/

/* Declare C function/method fct, having docstring docstr; may use vargargs */
#define Py_C_Function(fct,docstr) \
        static char fct##_docstring[] = docstr;\
        static PyObject *fct(PyObject *self, PyObject *args)

/* Declare C function/method fct, having keywords keywordsarray and a
   docstring docstr; may use vargargs & keywords */
#define Py_C_Function_WithKeywords(fct,docstr) \
        static char fct##_docstring[] = docstr;\
        static PyObject *fct(PyObject *self, PyObject *args, PyObject *kw)

/* these declare: self -- instance pointer for methods, NULL for functions
                  args -- argument tuple
		  kw   -- keywords dict (if applicable)
   plus as statics:
                  <function name>_docstring -- the docstring as given
		  <function name>_keywords  -- the keyword array as given

   note: use the Py_GetArg macros for functions without keywords,
             and Py_KeywordGetArg macros for functions with keywords
*/

/* Add a C function/method cname to the module dict as pyname; no
   doc-string */
#define Py_MethodListEntryAny(pyname,cname) {pyname,(PyCFunction)cname,METH_VARARGS}

/* Add a C function/method cname to the module dict as pyname; the
   function can use varargs */
#define Py_MethodListEntry(pyname,cname) {pyname,(PyCFunction)cname,METH_VARARGS,cname##_docstring}

/* Add a C function/method cname to the module dict as pyname; the
   function takes no args */
#define Py_MethodListEntryNoArgs(pyname,cname) {pyname,(PyCFunction)cname,0,cname##_docstring}

/* Add a C function/method cname to the module dict as pyname; the
   function takes one argument: the object is passed in directly
   (without wrapping it into a tuple first), i.e. don't use
   the Py_GetArg-macros or PyArg_ParseTuple(). */
#define Py_MethodListEntrySingleArg(pyname,cname) {pyname,(PyCFunction)cname,0,cname##_docstring}

/* Add a C function/method that uses keywords to the module dict */
#define Py_MethodWithKeywordsListEntry(pyname,cname) {pyname,(PyCFunction)cname,METH_VARARGS | METH_KEYWORDS,cname##_docstring}


/*** Macro for Get/Setattr slot methods ***/

/* This assumes that name is a constant; the strcmp function is only
   called in case the attribute name length exceeds 10 characters and
   the first 10 characters match */
#define Py_WantAttr(var,name)						\
     (var[0] == name[0] &&						\
      (var[0] == 0 ||							\
       (sizeof(name) >= 1 && var[1] == name[1] &&			\
	(var[1] == 0 ||							\
	 (sizeof(name) >= 2 && var[2] == name[2] &&			\
	  (var[2] == 0 ||						\
	   (sizeof(name) >= 3 && var[3] == name[3] &&			\
	    (var[3] == 0 ||						\
	     (sizeof(name) >= 4 && var[4] == name[4] &&			\
	      (var[4] == 0 ||						\
	       (sizeof(name) >= 5 && var[5] == name[5] &&		\
		(var[5] == 0 ||						\
		 (sizeof(name) >= 6 && var[6] == name[6] &&		\
		  (var[6] == 0 ||					\
		   (sizeof(name) >= 7 && var[7] == name[7] &&		\
		    (var[7] == 0 ||					\
		     (sizeof(name) >= 8 && var[8] == name[8] &&		\
		      (var[8] == 0 ||					\
		       (sizeof(name) >= 9 && var[9] == name[9] &&	\
			(var[9] == 0 || 				\
			 (sizeof(name) >= 10 && 			\
			  strcmp(&var[10],&name[10]) == 0		\
			 )))))))))))))))))))))

/*** Some useful macros for text handling ***/

/* Check a given textslice and apply the usual rules for negative indices */
#define Py_CheckSlice(textobj,start,stop) {		\
	    if (stop > PyString_GET_SIZE(textobj))	\
		stop = PyString_GET_SIZE(textobj);	\
	    else {					\
		if (stop < 0)				\
		    stop += PyString_GET_SIZE(textobj);	\
		if (stop < 0)				\
		    stop = 0;				\
	    }						\
	    if (start < 0) {				\
		start += PyString_GET_SIZE(textobj);	\
		if (start < 0)				\
		    start = 0;				\
	    }						\
	    if (stop < start)				\
		start = stop;				\
	}


/*** Helper for startup type object initialization ***/

#define PyType_Init(x) x.ob_type = &PyType_Type

/* EOF */
#endif

